/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SERVER_H
#define SERVER_H

#include <intercom/events.h>
#include <intercom/common.h>
#include <intercom/data_stream_splitter.h>
#include <olcutils/slist.h>
#include <olcutils/refcnt.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file server.h
    \brief IP/unix domain socket server

    \addtogroup server_api Server
    @{
 */


/*! maximum length to which the queue of pending connections for a listened socket may grow */
#define SOCK_LISTEN_BACKLOG        5


struct icom_s_connections;

/*! state data for socket accept handler */
typedef struct {
  t_icom_sock_addr            addr;                         /*!< socket API address specification */
  socklen_t                   addr_len;                     /*!< length of socket address */
  int                         sock_family;                  /*!< socket family, unix domain or IP */
  int                         fd;                           /*!< socket accept file descriptor */
  slist_t*                    server_list;                  /*!< list of all mainted server connections */
  int                         max_connections;              /*!< maximum allowed server connections */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
  pthread_t                   handler;                      /*!< connection accept handler thread */
  t_icom_events*              p_events;                     /*!< reference to event object */
  struct s_icom_connections*  p_connections;                /*!< back reference to all server connections */
  void*                       p_user_ctx;                   /*!< reference to user data context */
  void*                       p_server_state;               /*!< reference to global server state */
} t_icom_sock_listener;


/*! state data for each server's connection */
typedef struct s_sock_server {
  t_icom_sock_listener*       listener;                     /*!< back reference to connection listener */
  pthread_t                   handler;                      /*!< server socket handler */
  int                         fd;                           /*!< file descript for connection to r/w data */
  t_icom_evt*                 p_evt;                        /*!< next processed event */
  t_icom_data_stream*         p_data_stream;                /*!< pointer to data stream splitter instance */
  void*                       p_data_chunk;                 /*!< pointer to temporary socket receive data chunk */
} t_icom_sock_server;


/*! state data for all maintained server connections */
typedef struct s_icom_connections {
  slist_t*                    p_lst_listeners;              /*!< list for all socket listeners */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
} t_icom_connections;



/*! declaration of a new server connection to be created */
typedef struct {
  t_icom_addr_decl            addr;                         /*!< unix domain or IP address */
  int                         max_connections;              /*!< number of allowed connections */
} t_icom_server_decl;


/*! sever side (convenient) state object, provides all information */
typedef struct {
  const t_icom_server_decl*   decl_table;                   /*!< reference to input address spec */
  t_icom_events*              p_events;                     /*!< reference to event object */
  bool                        p_events_needs_to_released;   /*!< true, when created internally */
  t_icom_connections*         p_connections;                /*!< back reference to connections list object */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
} t_icom_server_state;


/*!
 * server side message send back to requesting process from call back function
 *
 * \param p_evt pointer to event object retrieved from call back function
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_reply_to_sender( t_icom_evt* p_evt, const char* data, const int len );


/*!
 * server side message send back to all processes at requesting address from call back function
 *
 * \param p_evt pointer to event object retrieved from call back function
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_reply_to_address( t_icom_evt* p_evt, const char* data, const int len );


/*!
 * unsolicited broadcast message to all registered connections
 *
 * \param p_server_state pointer to socker server global context
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_broadcast_to_all( t_icom_server_state* p_server_state, const char* data, const int len );


/*!
 * server side message send back to all registered connections from call back function
 *
 * \param p_evt pointer to event object retrieved from call back function
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_reply_to_all_connections( t_icom_evt* p_evt, const char* data, const int len );



/*!
 * kill or working tasks and release state data of a server handler
 *
 * \param p pointer to server state data to be released
 */
void icom_kill_server_handlers( t_icom_server_state* p );


/*!
 * create connection handler for a list of server address
 *
 * This function works similar to \ref icom_create_server_handlers but
 * on  constrast to  that it  does  not create  a new  event pool  but
 * delivers  its data  to  an  existing one  instead.  This allows  to
 * forward  traffic from  client and  server connections  to the  same
 * event pool.
 *
 * \param decl_table reference to input address spec
 * \param decl_table_len number of elements in address spec table
 * \param p_events pointer to an existing event pool, must be created externally
 * \param p_user_ctx reference to user context data
 * \return pointer to the newly instantiated object or NULL in case of error
 */
t_icom_server_state* icom_create_server_handlers_with_event_pool(
  const t_icom_server_decl decl_table[],
  const int decl_table_len,
  t_icom_events* p_events,
  void* p_user_ctx
  );


/*!
 * create connection handler for a list of server address and an event handler
 *
 * The function creates  a set of socket listeners  to accept incoming
 * IP or unix domoin socket  connections. Futhermore an internal event
 * managing task  is instantiated  where all server  processes deliver
 * their data to.  Finally a callback function is  registered which is
 * invoked whenever a message from a client connection is received.
 *
 * \param decl_table reference to input address spec
 * \param decl_table_len number of elements in address spec table
 * \param max_evt_data_size maximum event size in octets
 * \param pool_size number of elementes in the event pool
 * \param p_evt_cb callback function to be invoked when data is received
 * \param p_user_ctx reference to user context data
 * \return pointer to the newly instantiated object or NULL in case of error
 */
t_icom_server_state* icom_create_server_handlers(
  const t_icom_server_decl decl_table[],
  const int decl_table_len,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx
  );

/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SERVER_H */
