/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <unistd.h>
#include <string.h>

#include <olcutils/alloc.h>
#include <olcutils/refcnt.h>
#include <intercom/events.h>
#include <intercom/log.h>
#include <intercom/server.h>


void icom_release_evt( t_icom_evt* p )
{
  if( p ) {
    cul_free( p->p_data );
    cul_free( p );
  }
}

t_icom_evt* icom_create_evt( const int max_data_size )
{
  t_icom_evt* p = (t_icom_evt *) cul_malloc( sizeof(t_icom_evt) );

  if( p ) {
    p->max_data_size = max_data_size;
    p->p_data = (char *) cul_malloc( max_data_size );
    if( p->p_data == NULL ) {
      cul_free( p );
      p = NULL;
    }
  }

  return p;
}


void icom_release_event_pool( t_icom_events* p )
{
  if( p )
  {
    t_icom_evt* p_evt;

    while( !clist_is_empty( & p->pool ) ) {
      p_evt = (t_icom_evt *)clist_remove_head( & p->pool );
      icom_release_evt( p_evt );
    }

    while( !clist_is_empty( & p->ready_list ) ) {
      p_evt = (t_icom_evt *)clist_remove_head( & p->ready_list );
      icom_release_evt( p_evt );
    }

    pthread_cond_destroy( & p->signal );
    pthread_mutex_destroy( & p->mutex );
    cul_free( p );
  }
}

t_icom_events* icom_create_event_pool( const int max_evt_data_size, const int pool_size )
{
  t_icom_events* p;
  t_icom_evt* p_evt;
  int i;

  p = (t_icom_events *) cul_malloc( sizeof(t_icom_events) );
  if( p == NULL )
    return NULL;

  memset( p, 0, sizeof(t_icom_events) );

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    icom_error( "could initialize event pool mutex error\n");
    cul_free( p );
    return NULL;
  }

  if( pthread_cond_init( & p->signal, 0 ) != 0 ) {
    icom_error( "could not initialize evt pool conditional error!\n" );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    return NULL;
  }

  /* initialize pool an ready ring buffers */
  p->pool_size = pool_size;
  clist_init( & p->pool );
  clist_init( & p->ready_list );
  for (i = 0; i < pool_size; i++)
  {
    /* p->evtBlock[i].pData = & p->pPcmDataHeap[ i * DATA_SIZE ]; */
    p_evt = icom_create_evt( max_evt_data_size );
    if( p_evt ) {
      clist_insert_tail( & p->pool, (t_clist *) p_evt );
    }
    else {
      icom_release_event_pool( p );
      return NULL;
    }
  }

  return p;
}


void* icom_event_handler( void* pCtx )
{
  t_icom_events* p = pCtx;

  /* call state event loop  */
  while( 1 ) {
    pthread_mutex_lock( & p->mutex );
    do {
      if( clist_is_empty( & p->ready_list ) ) {
        /* avoid race condition by emptying complete list.
           So start next blocking operation exclusively when list is empty */
        pthread_cond_wait( & p->signal, & p->mutex );
      }
    } while( clist_is_empty( & p->ready_list ) ); // handle spurious wakeups
    p->p_evt = (t_icom_evt*)clist_remove_head( & p->ready_list );
    pthread_mutex_unlock( & p->mutex );

    /* do callback */
    p->p_evt_cb( p->p_evt );

    pthread_mutex_lock( & p->mutex );
    clist_insert_tail( & p->pool, & p->p_evt->node);
    p->p_evt = NULL;
    pthread_mutex_unlock( & p->mutex );
  }

  return p;
}


void kill_icom_event_handler( t_icom_events* p ) {
  pthread_cancel( p->handler );
  usleep( 100000 ); /* bloody hack but hopefully works */
  icom_release_event_pool( p );
}


t_icom_events* icom_create_event_handler(
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb )
{
  t_icom_events* p = icom_create_event_pool( max_evt_data_size, pool_size );
  int retcode;

  if( p ) {
    p->p_evt_cb = p_evt_cb;
    retcode = pthread_create( &p->handler, NULL, icom_event_handler, p );
    if( ! retcode )
      retcode = pthread_detach( p->handler );
    if( retcode ) {
      icom_error( "creation of event handler thread failed with error %d\n", retcode );
      icom_release_event_pool( p );
      p = NULL;
    }
  }

  return p;
}
