/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <getopt.h>
#include <unistd.h>
#include <stdint.h>
#include <signal.h>

#include <olcutils/alloc.h>
#include <olcutils/memtrace.h>
#include <olcutils/refstring.h>
#include <intercom/client.h>
#include <intercom/log.h>
#include <intercom/revision.h>

static int evt_cb_demo( t_icom_evt* p_evt )
{
  int errors = 0;
  const char* p_user_data = (char *) p_evt->p_user_ctx;

  if( p_evt->type == ICOM_EVT_CLIENT_DATA ) {
    printf( "---> received event: %s", p_evt->p_data );
    if( p_user_data )
      printf( ", user context data: %s\n", p_user_data );
    else
      printf( "\n" );
  }

  return errors;
}


int test_client( const char*addr, const uint16_t port )
{
  t_icom_client_conn_handler* p_client_conn_handler;
  t_icom_addr_decl addr_decl;
  char request_msg[80];
  char* user_data = "client user data";
  int retcode;

  const int request_period_us = 10000; /* 10ms */
  const int error_period_us = request_period_us * 10;
  const int max_requests = 1000;
  int request = 0;
  int max_errors = 10;

  printf("\n%s()\n", __func__ );

  if( port ) { /* network address */
    addr_decl.sock_family = AF_INET;
    addr_decl.port = port;
  } else { /* unix domain socket address */
    addr_decl.sock_family = AF_UNIX;
  }

  cstr_strlcpy( addr_decl.address, addr, sizeof(addr_decl.address) );

  p_client_conn_handler = icom_create_client_connection_handler( & addr_decl, 256, 100, evt_cb_demo, user_data );
  if( p_client_conn_handler == NULL ) {
    fprintf( stderr, "could not create client handler error!\n" );
    return -1;
  }

  /* send a lot of requests */
  while( request < max_requests ) {
    snprintf( request_msg, sizeof(request_msg), "icom client request %d\n", request );
    retcode = icom_client_handler_request( p_client_conn_handler, request_msg, strlen( request_msg ) );
    if( ! retcode ) {
      usleep( request_period_us );
    } else {
      fprintf( stderr, "transfer error occured in request %d\n", request );
      usleep( error_period_us );
      if( max_errors-- > 0 )
        continue;
      else
        break;
    }
    ++request;
  }


  printf("\tkill socket handlers ...\n" );
  icom_kill_client_connection_handler( p_client_conn_handler );

  printf("\tsocket handlers killed\n" );
  sleep( 3 );

  printf("\tsleeping for another 3 seconds\n");
  sleep( 3 );

  printf("%s()\t\t[OK]\n", __func__ );

  return 0;
}


static void help( const char* app_name )
{
  printf("Invocation: %s [ options ]\n\n", app_name );
  printf("Options:\n");
  printf("--address\n-a\n");
  printf("\tSever IP address or socket file to connect to\n\n");
  printf("--port\n-p\n");
  printf("\tSever IP port to connect to\n\n");
  printf("--help\n-h\n");
  printf("\tThis help screen.\n\n");
}


static int parse_args( int argc, char* argv[], char* addr, const int max_addr_len, uint16_t* p_port )
{
  int optindex, optchar;
  const struct option long_options[] =
  {
    { "help", no_argument, NULL, 'h' },
    { "address", required_argument, NULL, 'a' },
    { "port", required_argument, NULL, 'p' },
  };
  char* end;

  const long min_port = 1000;
  const long max_port = 65536;
  long port = 0;

  printf("Test client application for libicom providing event based Interprocess Communication for Dummies\n" );
  printf("written by Otto Linnemann\n");
  printf("Copyright 2016 GNU Lesser General Public Licence. All rights reserved.\n\n");

  while( ( optchar = getopt_long( argc, argv, "ha:p:", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'h':
      help( argv[0] );
      return -1;
      break;

    case 'a':
      cstr_strlcpy( addr, optarg, max_addr_len);
      break;

    case 'p':
      port = strtol( optarg, &end, 10);
      if( port < min_port || port > max_port ) {
        fprintf( stderr, "port must be in the range [%ld, %ld] error!\n", min_port, max_port );
        return -1;
      }
      break;

    default:
      fprintf( stderr, "input argument error!\n");
      return -1;
    }
  }

  if( ! strlen( addr ) ) {
    fprintf( stderr, "valid address must be specified error!\n" );
    return -1;
  }

  *p_port = (uint16_t)port;
  return 0;
}


void signal_handler( int sig_num )
{
  printf( "%s: received sigpipe (%d)\n", __func__, sig_num );
  exit( -1 );
}

int main( int argc, char* argv[] )
{
  char addr[108] = { '\0' };
  uint16_t port;
  int result = 0;
  cul_allocstat_t allocstat;
  FILE* free_log_fp = NULL;

  if( signal( SIGPIPE, signal_handler ) == SIG_ERR )
    fprintf( stderr, "Could not register SIGPIPE error!\n");

  if( parse_args( argc, argv, addr, sizeof(addr), &port ) )
    return -1;

  icom_log_init();
  printf( "Connecting test client for libintercom revision %s to address: %s, [port: %u]... \n\n",
          g_icomlib_revision, addr, port );

  memtrace_enable();

  if( ! result )
    result = test_client( addr, port );

  memtrace_disable();


  allocstat = get_allocstat();
  printf("\n\n");
  printf("Memory Allocation Statistics in cutillib functions\n");
  printf("--------------------------------------------------\n");
  printf("       number of open allocations: %ld\n", allocstat.nr_allocs );
  printf("     number of failed allocations: %ld\n", allocstat.nr_allocs_failed );
  printf("  still allocated memory in bytes: %ld\n", allocstat.mem_allocated );
  printf("maximum allocated memory in bytes: %ld\n", allocstat.max_allocated );
  printf("\n");

  if( allocstat.nr_allocs != 0 || allocstat.mem_allocated != 0 )
  {
    fprintf( stderr, "ATTENTION: Some memory is leaking out, configure with memcheck option and check for root cause!\n\n");
    result = -1;
  }

  memtrace_print_log( stdout );

  free_log_fp = fopen("/tmp/freelog.txt", "w" );
  if( free_log_fp != NULL )
  {
    memtrace_print_free_log( free_log_fp );
    fclose( free_log_fp );
  }

  icom_log_release();
  return result;
}
