/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <utils.h>
#include <intercom/log.h>
#include <config.h>
#include <errno.h>

/*! create a new list and omit element data */
slist_t* clone_slist_without_elem( const slist_t* l, const void* data )
{
  slist_t* nl = slist_alloc();

  if( nl ) {
    while( l && l->data ) {
      if( l->data != data ) {
        nl = slist_prepend( nl, l->data );
      } else {
      }
      l = l->next;
    }
  }
  return nl;
}

void slist_print( slist_t* l )
{
  while( l ) {
    printf(" ====> list element %lx, data: %lx\n", (long)l, (long)l->data );
    l = l->next;
  }

  printf("\n\n");
}

t_icom_evt* create_new_event( t_icom_events* p_events )
{
  t_icom_evt* p_evt;

  pthread_mutex_lock( & p_events->mutex );
  if ( !clist_is_empty( & p_events->pool ) ) {
    p_evt = (t_icom_evt*)clist_remove_head( & p_events->pool );
  }
  else {
    icom_error("event queue overflow error, overwriting existing events\n");
    p_evt = (t_icom_evt*)clist_remove_head( & p_events->ready_list );
  }
  pthread_mutex_unlock( & p_events->mutex );

  return p_evt;
}

void send_event( t_icom_events* p_events, t_icom_evt* p_evt )
{
  pthread_mutex_lock( & p_events->mutex );
  clist_insert_tail( & p_events->ready_list, & p_evt->node );
  p_evt = NULL;
  pthread_cond_signal( & p_events->signal );
  pthread_mutex_unlock( & p_events->mutex );
}


int get_evt_max_data_size( t_icom_events* p_events )
{
  int max_data_size;
  t_icom_evt* p_evt;

  pthread_mutex_lock( & p_events->mutex );
  p_evt = (t_icom_evt*)clist_get_tail( & p_events->pool );
  max_data_size = p_evt->max_data_size;
  pthread_mutex_unlock( & p_events->mutex );

  return max_data_size;
}


/*
 * be aware that send does not ALWAYS send length bytes
 * so we better use a wrapper (last minute change)
 */
static int send_all(int socket, const void *buffer, size_t length, int flags)
{
  ssize_t n;
  size_t bytes_to_write = length;
  const char *p = buffer;

  while (bytes_to_write > 0)
  {
    n = send(socket, p, bytes_to_write, flags);

    if ( n <= 0 ) {
      if( errno == EAGAIN  ||  errno == EINTR ) {
        icom_error( "%s, %d: send interrupt, retry: %d, %s\n", __func__, __LINE__, errno, strerror(errno) );
      } else {
        icom_error( "%s, %d: send failed: %d, %s\n", __func__, __LINE__, errno, strerror(errno) );
        return -1;
      }
    } else {
      p += n;
      bytes_to_write -= n;
    }
  }

  return length;
}

ssize_t send_with_delim( int socket, const void *buffer, size_t length, int flags )
{
#ifdef ENABLE_ADDING_MISSING_MSG_DELIM
  ssize_t sent;
  const char* delim_bytes = MESSAGE_DELIM;
  const int delim_bytes_len = strlen( MESSAGE_DELIM );
  int sent_delim;

  sent = send_all( socket, buffer, length, flags );
  if( sent == length &&
      memcmp( buffer + length-delim_bytes_len, delim_bytes, delim_bytes_len ) )
  {
    sent_delim = send_all( socket, delim_bytes, delim_bytes_len, flags );
    if( sent_delim != delim_bytes_len ) {
      sent = -1;
    }
  }

  return sent;
#else
  return send_all( socket, buffer, length, flags );
#endif /* ENABLE_ADDING_MISSING_MSG_DELIM */
}
