/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef ALLOC_H
#define ALLOC_H

#include <stdlib.h>
#include <pthread.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file alloc.h
    \brief memory allocation and deallocation
    \defgroup memory_wrapper Memory Allocation Wrapper Functions for Leak Analyses
    \ingroup memory_api wrapper
    @{

    ## Track Memory Allocations within Application and Utilities Library

    Memory tracing  by overloading  malloc and  free does  incorporate not  only the
    application under  development but all  linked libraries  as well. When  such an
    external library maintains a lot of external state which must not necessarily be
    freed at any time the application runs, the list of non-freed memory as a result
    by just overloading and tracking malloc and free invocations can get very long.

    As an  alternative we  provide another  instrumentation which  is just  based on
    macro  overloading. In  fact  it is  the  recommended way  of  ensuring that  an
    application respectively the  algorithms in this utility library is  free of any
    memory leaks. The  implementation is expected to free  all dynamically allocated
    memory by  calling cul_free()  before it terminates.  These memory  chunks shall
    have  been previously acquired by  calling cul_malloc()  or cul_realloc().  Just
    before termination the  table of all memory allocations in  terms of file, line,
    function and size of the leaking chunk can be printed out.

    There are two different implementations to track non-freed allocations based on:

    1. a flat array via the configure option 'alloctrace'
    2. a hash map via the configure option 'lifetime-alloctrace'

    ### Tracing Memory Blocks via a Flat array (alloctrace)

    The  first configuration  option is  required  when using  functionality of  the
    present utility  library. It uses  a much  more primitive and inefficient memory
    trace table  but it includes invocations  of all cutils library  functions, too.
    The library must  be compiled with the configuration option  'alloctrace' in the
    following way to enable the full alloc checker:

        ./configure --enable-alloctrace

    The  functional   test  implementation  within  the   source  file  test_main.c
    illustrates  how the  memory  allocation can  be printed  out  to the  standard
    console. Be  aware that  the alloctrace option  involves a  significant runtime
    penalty and should be therefore disabled in production.


    ### Tracing Memory Blocks via Hashmap (lifetime-alloctrace)

    Since we use the 'ideal hash tries' algorithm implemented in the present library
    for the  second approach, we  cannot instrument  the provided functions  of this
    library  e.g. the  hash map  implementation itself  in this  way. So  the second
    approach  is exclusively  covering memory  allocations which  are not  using any
    functionality provided  by this library  but given the  look up efficiency  of a
    hash table  it can be  beneficial when  it comes to  the debugging of  a complex
    application where  potentially thousands  of explicitly allocated  memory blocks
    are in use and the  functionality of the present cutils library is not used. All
    invocation of  malloc, realloc and  free of the  application under test  must be
    of  course  replaced with  their  counterparts  cul_malloc(), cul_realloc()  and
    cul_free().

    The second approach offers additional features which can be particularly helpful
    in  the discovery  of memory  leaks in  cases where  a graceful  shutdown of  an
    application is not available.

    - storing of the application runtime timestamp for every allocation operation
    - allowing to automatically write the list of non-freed memory blocks to a file

    For   debugging  explicitly   allocated  memory   blocks  via   cul_malloc()  or
    cul_realloc() run the following recommended memory debug configuration:

        ./configure --enable-lifetime-alloctrace

    By  default  the  library  writes  a list  of  all  non-free  memory  allocation
    blocks after  one hour to the  file '/tmp/<executable-name>_alloctrace'  in this
    configuration. The  first entry of this  list is  the timestamp  reflecting when
    the  associated memory  block  was  allocated. The  trigger  period for  writing
    the  allocation  list  can  be  changed  via  a  control  file  under  the  name
    '/tmp/<executable-name>_alloctrace_control'. In the  following example we assume
    that the application sscd2 is started. Then the command

        echo 30 > /tmp/sscd2_alloctrace_conrol

    changes the trigger time to 30 seconds. When sscd2 is started after this control
    parameter  has been  set it  automatically  writes all  non-freed memory  blocks
    allocated with  cul_malloc() or cul_realloc() to  /tmp/sscd2_alloctrace after 30
    seconds runtime.  The expire condition  is checked within calls  to cul_malloc()
    and  cul_realloc(). The  following command  shows the  list sorted  in ascending
    order by timestamp:

        cat /tmp/sscd2_alloctrace | sort -n

    This allows to  detect potential memory leakages even in  cases where a graceful
    shutdown  of an  application  is  not available.  Typically  memory blocks  with
    program lifetime are allocated during  the program startup initialization phase.
    Memory  blocks  allocated later  during typical  runtime  operations  should  be
    released after a while.  As a consequence the  list of  open allocations  should
    exclusively  come  with the  early  lifetime  allocation  and the  late  runtime
    allocations which  have not been freed  yet. Everything in between is suspicious
    and  should be  analyzed more  closely. But  this should  be relatively straight
    forward since  for every  allocation the  source file, the  line number  and the
    function where the allocation took place  are integrated in the allocation list.
    The allocation  trace is also  written whenever the function  get_allocstat() is
    invoked which is recommended to be done  as last instruction when all memory has
    been released.


    ## Check for Out of Bounds Memory Access Violations

    Fortunately memory  access violation do  occur more  rarely but the  analysis of
    these  errors  can  get  quite  difficult.  A  common  approach  is  to  prepend
    respectively  append  a  prefix  and   a  postfix  sequence  to  each  allocated
    memory  block.  Usually  wrong  index   calculations  cause  more  bytes  to  be
    written than  actually allocated  which causes  the prefix  or postfix  block to
    be  overwritten.  This  is  checked  in  every  memory  release  operation  such
    as  cul_free()  or  cul_realloc()  or  at any  time  by  invoking  the  function
    cul_check_for_oob_violation(). To enable oob checks use the following command:

        ./configure --enable-alloctrace --enable-oobcheck --enable-oobassert

    or

        ./configure --enable-lifetime-alloctrace --enable-oobcheck --enable-oobassert
*/

/*! debug information holding __func__ and __LINE__ tags*/
typedef struct {
  const char* func;       /*!< pointer to __func__ macro */
  const char* file;       /*!< pointer to __FILE__ macro */
  int line;               /*!< content of __LINE__ macro */
} cul_dbg_info_t;


/*! key value pair of memory address and allocation debug info */
typedef struct {
  void*      addr;        /*!< address returned by cul_malloc */
  size_t     size;        /*!< requested and non-freed size allocated at given address */
  cul_dbg_info_t dbg;     /*!< associated debug information (function, line number) */
} cul_addr_dbg_info_t;

typedef struct {
  cul_addr_dbg_info_t* addr_table;      /*!< pointer, debug info hash map (linear) */
  long                 addr_table_size; /*!< maximum size of add_table */
  long                 addr_table_head; /*!< current head index of addr_table starting at 0 */
  pthread_mutex_t      mutex;           /*!< mutex to rule thread access */
} cul_alloc_trace_t;

/*! statistic information about the memory consumption of the library */
typedef struct {
  long nr_allocs;         /*!< number of open (unfreed) allocation operations */
  long nr_allocs_failed;  /*!< number of failed allocation operations for defered out of memory detection */
  long mem_allocated;     /*!< number of open (unfreed) memory in bytes */
  long max_allocated;     /*!< maximum allocated memory space used by this library in bytes */

  cul_alloc_trace_t alloc_trace; /*!< allocation tracing (function, line number of each alloc) */
} cul_allocstat_t;


/*!
 * \brief retrieve memory allocation statistics
 *
 * \return statistics about currently and at maximum allocated memory space
 */
cul_allocstat_t get_allocstat( void );


/*!
 * \brief allocate statistically tracked memory
 * wrapper to stdlib function malloc with same invocation scheme
 *
 * \param size size in bytes to be reserved
 * \return pointer to allocated memory area or NULL in case allocation failed
 */
#ifdef OLCUTILS
#define cul_malloc( size ) _cul_malloc( size, __func__, __FILE__, __LINE__, 0 )
#else
#define cul_malloc( size ) _cul_malloc( size, __func__, __FILE__, __LINE__, 1 )
#endif /* #ifdef OLCUTILS */
void* _cul_malloc( size_t size, const char* func, const char* file, const int line, const int ext );


/*!
 * \brief reallocate statistically tracked memory
 * wrapper to stdlib function realloc with same invocation scheme
 *
 * \param size new size in bytes to be reserved
 * \param ptr pointer to alloready allocated area or NULL if nothing has been allocated yet
 * \return pointer to allocated memory area or NULL in case allocation failed
 */
#ifdef OLCUTILS
#define cul_realloc( ptr, size ) _cul_realloc( ptr, size, __func__, __FILE__, __LINE__, 0 )
#else
#define cul_realloc( ptr, size ) _cul_realloc( ptr, size, __func__, __FILE__, __LINE__, 1 )
#endif /* #ifdef OLCUTILS */
void* _cul_realloc( void* ptr, size_t size, const char* func, const char* file, const int line, const int ext);


/*!
 * \brief free statistically tracked memory
 * wrapper to stdlib function free with same invocation scheme
 *
 * \param ptr pointer to memory area to be freed.
 */
#ifdef OLCUTILS
#define cul_free( ptr ) _cul_free( ptr, __func__, __FILE__, __LINE__, 0 )
#else
#define cul_free( ptr ) _cul_free( ptr, __func__, __FILE__, __LINE__, 1 )
#endif /* #ifdef OLCUTILS */
void _cul_free( void* ptr, const char* func, const char* file, const int line, const int ext  );


#define CUL_PREFIX_VIOLATED_FLAG   0x1 /*!< prefix of memory allocation block overwritten */
#define CUL_POSTFIX_VIOLATED_FLAG  0x2 /*!< postfix of memory allocation block overwritten */


/*!
 * \brief check prefix and postfix of memory block
 *
 * The  build feature  'oobcheck' must  be enabled  otherwise the  function returns
 * alays  0.  Be   aware  that  this  comes  with  significant   run  time  penalty
 * so  it  is  recommended  to  always  disable  it  for  production  builds.  When
 * enabled  the function  returns  a bitbask  with CUL_PREFIX_VIOLATED_FLAG  and/or
 * CUL_POSTFIX_VIOLATED_FLAG  for  memory  blocks   allocated  with  cul_malloc  or
 * cul_realloc in case  a violation occured. When the build  feature 'oobassert' is
 * present, the  current process is  terminated as soon  as an access  violation is
 * detected.
 *
 * \param ptr pointer to block whose pre-/postfix needs to be checked
 * \return bitbask of detected violations
 */
int cul_check_for_oob_violation( void* ptr );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef ALLOC_H */
