/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef CLIST_H
#define CLIST_H


#ifdef __cplusplus
extern "C" {
#endif

/*! \file clist.h
    \brief circular linked lists

    \addtogroup circular_list_api
    @{
 */

typedef struct s_clist t_clist;

/*!
 * data structure for doubly linked lists
 */
typedef struct s_clist {
  t_clist* prev;        /*!< pointer to previous element */
  t_clist* next;        /*!< pointer to next element */
} t_clist;


/*!
 * initialze a new empty circular list structure
 i*
 * \param p pointer to list element
 */
void clist_init( t_clist* p );


/*!
 * check whether circular list is empty
 *
 * \param p pointer to list element
 * \return 0 if not empty, otherwise positive (bool) value
 */
int clist_is_empty( t_clist* p );


/*!
 * insert new element at head of the list
 *
 * \param p pointer to clist structure
 * \param e pointer to list element to be inserted
 */
void clist_insert_head( t_clist* p, t_clist* e );


/*!
 * insert new element at tail of the list
 *
 * \param p pointer to clist structure
 * \param e pointer to list element to be inserted
 */
void clist_insert_tail( t_clist* p, t_clist* e );


/*!
 * removes element from head of the list
 *
 * \param p pointer to clist structure
 * \return e pointer to removed list element
 */
t_clist* clist_remove_head( t_clist* p );


/*!
 * removes element from tail of the list
 *
 * \param p pointer to clist structure
 * \return e pointer to removed list element
 */
t_clist* clist_remove_tail( t_clist* p );


/*!
 * get head element from circular list
 *
 * \param p pointer to clist structure
 * \return pointer to head element
 */
t_clist* clist_get_head( t_clist* p );


/*!
 * get tail element from circular list
 *
 * \param p pointer to clist structure
 * \return pointer to tail element
 */
t_clist* clist_get_tail( t_clist* p );


/*!
 * print all elements of list structure
 *
 * This is mainly for debugging purposes.
 *
 * \param first pointer to clist structure
 * \param print_element_cb callback function for printing element
 */
void clist_print_elements( t_clist* first, void (*print_element_cb)(t_clist*) );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef CLIST_H */
