/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef HASHMAP_H
#define HASHMAP_H

#include <stdint.h>
#include <olcutils/lambda.h>
#include <olcutils/slist.h>
#include <olcutils/refstring.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file hashmap.h
    \brief ideal hash tries

    \addtogroup hashmap_api Ideal Hash Tries
    \ingroup functional_programming_api
    @{

    API functions to a very efficient implementation for hash maps,
    refer to section \ref ideal_hash_tries "Ideal Hash Tries" for
    more information how to use them.
 */


/*!
 * hash data type used as key, currently a 32 bit unsigned integer value
 */
typedef uint32_t hm_hash_t;


/*!
 * count bit population within an integer (32 bit)
 * counts number of least significant bits set to 1
 *
 * this is just for illustration and checking purposes
 */
int CTPop_check( hm_hash_t i );


/*!
 * count bit population within an integer (32 bit)
 * counts number of least significant bits set to 1
 *
 * taken from "Ideal Hash Trees", by Phil Bagwell
 * http://lampwww.epfl.ch/papers/idealhashtrees.pdf
 */
int CTPop( hm_hash_t map );


/*! key-value pair (leaf) stores actual information within hash tree */
typedef struct hm_leaf_node_struct {
  hm_hash_t hash; /*!< the hash key */
  void* val; /*!< a pointer to the associated value */
} hm_leaf_node_t;


/*! a node within the hash can either be meta data or an actual key-value pair */
typedef enum {
  bitmap_index_node_type, /*!< node is an bit map index node */
  leaf_node_type /*!< node is a leaf node of type hm_leaf_node_t */
} hm_node_des_t;



struct hm_inode_struct;

/*! bit map index node type */
typedef struct hm_bitmap_indexed_node_struct {
  hm_hash_t bitmap; /*!< the bit map representing assigned elements in this node */
  struct hm_inode_struct* nodes; /*!< pointer to an array of the assigned nodes */
  int shift; /*!< number of right shifts assigned to this node */
} hm_t;


/*! ideal hash tries node */
typedef struct hm_inode_struct {
  /*! variant which carries either a bitmap index node or a leaf node */
  union hm_inode_union {
    struct hm_bitmap_indexed_node_struct inode; /*!< bit map index node */
    struct hm_leaf_node_struct lnode; /*!< leaf node */
  } u; /*!< either a bitmap index node or a leaf node */
  hm_node_des_t typ; /*!< type of node */
} hm_inode_t;


/*!
 * creates a new empty hash map
 * \return pointer to hash map or NULL in case of error (out of memory)
 */
hm_t* hm_alloc( void );


/*!
 * associates a (new) hash value to a given hash key
 *
 * Creates a new key-value pair or overwrites the associated value
 * if the key already exsists. The operation is O(1).
 *
 * \param this pointer to hash map, use hm_alloc() to create one
 * \param shift shift level for this invocation, invoke always with 0
 * \param hash hash key, use e.g. string_hash() or cstring_hash() to generate it
 * \param val pointer to value object
 * \return pointer to index node where value is stored or NULL in case of error
 */
hm_t* hm_assoc( hm_t* this, int shift, hm_hash_t hash, void* val );


/*!
 * dissociate hash value for given key
 *
 * removes a key-value pair from a given hash map. Note that the
 * value object is not freed. The operation is O(1).
 *
 * \param this pointer to hash map, use hm_alloc() to create one
 * \param shift shift level for this invocation, invoke always with 0
 * \param hash hash key to be removed, use e.g. string_hash() or cstring_hash()
 *        to generate it
 * \return pointer to index node where value was stored or NULL if
 *         hash key was not found.
 */
hm_t* hm_dissoc( hm_t* this, int shift, hm_hash_t hash );


/*!
 * find hash value for given key
 *
 * The function retrieves the key-value pair node assigned
 * to a given hash key. The operation is O(1).
 *
 * \param this pointer to hash map
 * \param hash hash key to be found, use string_hash() or cstring_hash()
 *        to generate it
 * \return pointer to key-value structure of type hm_leaf_node_t or NULL
 *         if not found
 */
hm_leaf_node_t* hm_find( hm_t* this, hm_hash_t hash );


/*!
 * apply function f to hash map
 *
 * Invoke function to each key-value pair of the given hash map.
 * f takes two void pointers as arguments, the first is the key
 * of type hm_hash_t, the second needs to be casted to the user
 * data type. hm_doseq does not alter the hash map and is mostly
 * not a pure function that means there are side effects I/O
 * or changed memory.
 *
 * \param this pointer to hash map
 * \param level recursion level, invoke always with 0
 * \param f lambda function with two argument pointers (key, value)
 */
void hm_doseq( hm_t* this, int level, lambda2_t f );


/*!
 * Transform hash map to singly linked list
 *
 * Transform the given hash map to a singly linked list. There
 * no guarantees about the output order within the list. Each
 * list element is of type hm_leaf_node_t representing a key-
 * value pair.
 *
 * \param l pointer to list to append elements to,
 *        when NULL hm_slist creates a list
 * \param this hash map to be transformed
 * \return pointer to generated list with key-value pairs
 */
slist_t* hm_slist( slist_t* l, hm_t* this );


/*!
 * Release memory of hash map meta data
 *
 * Frees all memory used to represent meta data within the hash
 * but does not free memory used for the value objects within
 * the hash map.
 *
 * \param this pointer to hash map
 * \param level recursion level, invoke always with 0
 */
void hm_free( hm_t* this, int level );


/*!
 * Release memory of hash map meta and value data
 *
 * Frees all memory represented by given hash including value
 * objects. You have to provide a free operation which is
 * applied to each value stored within the hash.
 *
 * \param this pointer to hash map
 * \param level recursion level, invoke always with 0
 * \param free_op pointer to release function for value objects
 */
void hm_free_deep( hm_t* this, int level, lambda_t free_op );



/*! hash map which stores key as referenced string as well */
typedef struct {
  string_t* key; /*!< string object for key */
  void* val; /*!< value */
} hm_kv_t;


/*!
 * associates a (new) hash value to a given hash key string
 *
 * Creates a new key-value pair or overwrites the associated value
 * if the key already exsists. The operation is O(1). On contrast to
 * hm_assoc() this function stores the original key string together
 * with its associated value as an object of type hm_kv_t. This is
 * often useful when key strings need to be read out
 * from the hash data in example for parsers.
 *
 * \param this pointer to hash map, use hm_alloc() to create one
 * \param keystr length delemited string type which represents the key
 * \param val pointer to value object
 * \return pointer to index node where value is stored or NULL in case of error
 */
hm_t* hm_assoc_with_key( hm_t* this, string_t* keystr, void* val );


/*!
 * dissociate hash value for given key string
 *
 * removes a key-value pair from a given hash map. Note that the
 * value object is not freed. The operation is O(1). This function
 * works requires that the given hash map elements have been
 * exclusively stored by hm_assoc_with_key().
 *
 * \param this pointer to hash map
 * \param keystr length delemited string type which represents
 *        the key to be removed
 * \return pointer to index node where value was stored or NULL if
 *         hash key was not found.
 */
hm_t* hm_dissoc_with_key( hm_t* this, string_t* keystr );


/*!
 * find hash value for given key string
 *
 * The function retrieves the pointer to the value object assigned
 * to a given hash key string. The operation is O(1). This function
 * works requires that the given hash map elements have been
 * exclusively stored by hm_assoc_with_key().
 *
 * \param this pointer to hash map
 * \param keystr length delemited string type which represents
 *        the key to be found
 * \return pointer to value object for the given key or NULL
 *         when not found
 */
void* hm_find_val_for_key( hm_t* this, string_t* keystr );


/*!
 * Transform hash with value objects of hm_kv_t to singly linked list
 *
 * Transform the given hash map whose elements have been associated by
 * hm_assoc_with_key() to  a singly  linked list. There  no guarantees
 * about the  output order within  the list.  Each list element  is of
 * type hm_kv_t representing a pair of the key string and a pointer
 * to the value object.
 *
 * \param l pointer to list to append elements to,
 *        when NULL hm_slist creates a list
 * \param this hash map to be transformed
 * \return pointer to generated list with key-value pairs
 */
slist_t* hm_slist_with_keys( slist_t* l, hm_t* this );


/*!
 * Release memory of hash map meta data
 *
 * Frees all memory of hash map whose elements have been associated by
 * hm_assoc_with_key(). The function does not free memory used for the
 * value objects within the hash map.
 *
 * \param this pointer to hash map
 */
void hm_free_with_keys( hm_t* this );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef HASHMAP_H */
