/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef JSON_H
#define JSON_H

#include <stdio.h>
#include <olcutils/hashmap.h>
#include <olcutils/slist.h>
#include <olcutils/refstring.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file json.h
    \brief Parsing JSON Data

    \addtogroup json_api Parsing JSON Data
    \ingroup parsers
    @{

    Refer to \ref json_introduction the introductive section "Parsing of JSON Data"
    for basic information about JSON and the fundamentals cutils implements it.

    In the following we illustrate how to use the API for parsing JSON data. The
    given sample code snippets here do not include any error checking. Refer to
    the test function test_json.c for more complete usage scenario:

    Let's assume we have got some JSON data as character array

        char* c_data = "{ \"KEY1\" : 42,  \"KEY2\" : \"STRING VALUE\", }";

    We create a \ref string_datatyp "length delimited string datatyp" out of this
    and invoke the parser with that by the following lines:

        string_t* json_string = string_new_from( c_data );
        json_val_t* my_json = json_parse_string( json_string );

    Now we can look up all kinds of information from the parsed JSON data. First
    we check for any parser errors. If the my_json is error free we can look
    up the keys:

        found = json_errors( my_json );
        if( found )  {
          printf("parsing error %d occured\n", (int)found->u.error);
        }
        else {

           string_t* search_key1 = string_new_from( "KEY1" );
           string_t* search_key2 = string_new_from( "KEY2" );

           found = json_find( my_json, search_key1, 1 );
           if( found )
             printf("got value:  "); printf( "%lf\n", found->u.json_number );

           found = json_find( my_json, search_key2, 1 );
           if( found )
             printf("got value:  "); string_println( found->u.string );
        }

    In the above code you shall check that the type of the found entity matches
    your expection by comparing found->typ to expected typedefinition.

    Do not forget to release the memory correctly as in the given example:

        string_release( search_key1 );
        string_release( search_key2 );
        json_free( my_json );
*/


#define json_true  1   /*!< the JSON Boolean value true */
#define json_false 0   /*!< the JSON Boolean value false */

/*! flag  for declaring  one out  of the  five JSON  data types  plu Ones.
 *  additiona type is revered for parser errors. */
typedef enum {
  json_object,      /*!< indicates a JSON object mapped to cutils hash-map */
  json_array,       /*!< indicates a JSON array mapped to cutils singly linked list */
  json_number,      /*!< indicates a JSON number mapped to the C-typ double */
  json_boolean,     /*!< indicates a JSON boolean value mapped to C-typ int */
  json_string,      /*!< indicates a JSON string value mapped to cutils string_t */
  json_parse_error  /*!< indicates a parser error */
} json_val_typ;

typedef double json_number_t; /*!< the mapping from JSON to C number type */
typedef int json_boolean_t;   /*!< the mapping from JSON to C boolean type */

/*! enumeration of several common parse errors */
typedef enum {
  json_unbalanced_quotes,    /*!< one quote character (") is missing */
  json_unbalanced_braces,    /*!< one curly brace character ({}) is missing */
  json_unbalanced_brackets,  /*!< one bracket character ([]) is missing */
  json_unbalanced_colon,     /*!< a colon is missing or wrongly placed in JSON object */
  json_invalid_number,       /*!< JSON number is incorrectly formatted */
  json_unspecified_error     /*!< unknown error */
} json_parse_error_t;


/*! JSON value type which can be  string, number, object, array or true or
 *  false of typ boolean. The NULL is currently not supported and might be
 *  added in further versions. */
typedef struct {
  /*! variant which keeps any of the six JSON values */
  union {
    hm_t* object;              /*!< JSON objects are mapped to cutils hash maps */
    slist_t* array;            /*!< JSON arrays are mapped to cutils signly linked lists */
    json_number_t number;      /*!< JSON numbers are mapped to the C-type double */
    json_boolean_t boolean;    /*!< JSON booleans are integers in C with value 0 and 1 */
    string_t* string;          /*!< JSON strings are mapped to lengths delimeted cutils strings */
    json_parse_error_t error;  /*!< When a parsing error occurs the error value indicates its type */
  } u;
  json_val_typ typ;  /*!< indicates the type of JSON value stored in u */
} json_val_t;


/*!
 * Parsing of JSON data
 *
 * The data to  be parsed is provided as length  delimited cutils String.
 * Refer  to \ref  refstring_api  "API description  for length  delimited strings"
 * for more  detailed information how to  create them. The result  of the
 * parsing  process  is  a  recursively nested  data  structure  of  type
 * json_val_t. Its  key values can  be efficiently retrieved  by invoking
 * json_find(). Note that occurence of parsing errors needs to be checked
 * for by  invoking json_errors()  which will return specific information
 * about the malformatted object and error reason.
 *
 * \param s input string carrying JSON data
 * \return a pointer to a recursive data structure of type json_val_t
 */
json_val_t* json_parse_string( string_t* s );


/*!
 * Transform parsed JSON data back to string representation
 *
 * When parsed JSON data is directly manipulated which is not required in
 * most applications,  the function  json_emerge allows to  transform the
 * internal  cutils  representation  back  to  an  original  JSON  string
 * representation.
 *
 * \param fp file pointer where to write output result to. Use stdout for
          console
 * \param val pointer to JSON value
 */
void json_emerge( FILE* fp, json_val_t* val );


/*!
 * Release memory of parsed JSON data
 *
 * A call to json_free() releases all memory allocated by json_parse_string().
 *
 * \param val point to JSON data structure to be released.
 */
void json_free( json_val_t* val );


/*!
 * Search recursively for a given key within JSON data
 *
 * The core of JSON  data is the JSON object which is  a hash map storing
 * key-value pairs. This functions does  not only search within the given
 * JSON  object for  the given  key  but also  recursively retrieves  any
 * encapsulated objects up to the given number of recursion levels.
 *
 * \param val pointer to parsed JSON data
 * \param key to search for
 * \param max_levels number of maximum recursion levels
 * \return value object for the given key or NULL if nothing found
 */
json_val_t* json_find( json_val_t* val, string_t* key, int max_levels );


/*!
 * Checks for parsing errors within parsed JSON data
 *
 * This function will return the  first parsing error it discovers within
 * the given  parsed JSON input data.  In case NULL is  returned the JSON
 * parsed data  can be considered to  be valid JSON formatted.  Note that
 * cutils currently  does not fully comply  to all subtle details  of the
 * JSON format. So a NULL does  exclusively mean that json_parse was able
 * to deal with the given data.
 *
 * \param val pointer to parsed JSON data
 * \return defective json object which will come with an empty key and an
           enumeration value of type json_parse_error_t indicating the
           reason for this error or NULL if no errors have been detected
 */
json_val_t* json_errors( json_val_t* val );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef JSON_H */
