/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef MEMTRACE_H
#define MEMTRACE_H

#ifdef __cplusplus
extern "C" {
#endif

/*! \file memtrace.h
    \brief memory leakage detection
    \defgroup memory_overloading Overloading the memory management functions of the C-Library
    \ingroup memory_api
    @{

    ## Detecting Memory Leaks

    In C  basically all dynamic memory  management is based on  three functions from
    the standard C  library: malloc(), realloc() and free(). These  functions can be
    overloaded  on most  UNIX  systems by  replacements which  keep  track of  their
    invocations. The library needs be configured  and compiled with the build option
    "enable-memtrace"  to  activate the  aforementioned  overloading.  Whenever a  C
    program is  linked against libcutils, an invocation of malloc() in  example does
    not invoke the c-libraries malloc directly but instead invokes libcutils malloc.
    Memory tracing can be enabled and disabled by the function memtrace_enable() and
    memtrace_disalbe(). All invocation in between of  malloc are stored in a list of
    used  memory blocks.  Whenever  free()  is invoked  the  corresponding block  is
    removed from  the list. The  function memtrace_print_log() writes  all non-freed
    blocks to a given file descriptor, use stdout in order to write the output table
    directly to the console. Beneath address  and size a backtrace of all invocation
    queue  for the  corresponding call  to  malloc() shall  be given.  Make sure  to
    integrate debug options  and better avoid strong runtime  optimizations in order
    to achieve that, e.g. on Linux use the following command for configuration:

        ./configure --enable-memtrace CFLAGS="-ggdb -O1"

    ## Detecting Memory Access Violations

    The library can  be configured to keep  track of all freed memory  as well. This
    can be  helpful when searching for  memory access violations, especially  when a
    memory block  is freed too early  before it accessed from  another function. Use
    the  configuration option  enable-freetrace to  activate this  functionality. Be
    aware that this trace operation intentionally is a memory leak because this list
    is never freed. So ensure never to enable this option when releasing the library
    together with  an application.  Pay attention  to the fact  that this  option is
    dependend on  the option  memtrace. In example  use the  following configuration
    command to enable tracing of free operations:

        ./configure --enable-memtrace --enable-freetrace CFLAGS="-ggdb -O1"

    Afterwards  the API  function  memtrace_print_free_log() writes  a  list of  all
    invocations to free to given file descriptor.
 */


/* @cond */

#define MEMTRACE_MAX_BACKTRACE_ELEMENTS   20

/*  function pointer array for backtrace call hierarchy */
typedef struct backtrace_array_struct {
  int    size;
  void** p;
} backtrace_array_t;


/*  member data stored for each memory allocation */
typedef struct memblck_struct {
  void*              addr; /*  pointer to actual memory (what is return by malloc) */
  size_t             size; /*  size of the memory block referenced by address */
  backtrace_array_t  bt;   /*  array of function pointers with call hierarchy */
} memblck_t;


/*  linked list for tracing all memory allocations */
typedef struct mlist_struct {
  memblck_t            data;
  struct mlist_struct* next;
} mlist_t;


/*  linked list for tracing all memory free operations */
typedef struct flist_struct {
  memblck_t            data;
  backtrace_array_t    bt_free;
  struct flist_struct* next;
} flist_t;

/* @endcond */


/*!
 * enable tracing of allocator callbacks for leakage analysis
 *
 * The library needs to be appropriately configured for this. Refer to
 * the section
 * \ref memory_overloading "Detailed Description"
 * for more information.
 */
void memtrace_enable( void );


/*!
 * disable tracing of allocator callbacks for leakage analysis
 *
 * The library needs to be appropriately configured for this. Refer to
 * the section
 * \ref memory_overloading "Detailed Description"
 * for more information.
 */
void memtrace_disable( void );


/*!
 * print non-free memory chunks with allocator's backtrace
 *
 * The library needs to be appropriately configured for this. Refer to
 * the section
 * \ref memory_overloading "Detailed Description"
 * for more information.
 *
 * \param fp file pointer where logging information shall be written to
 * \return 0 in case of success otherwise negative error number.
 */
int memtrace_print_log( FILE* fp );


/*!
 * print all freed memory chunks with free's backtrace
 *
 * The library needs to be appropriately configured for this. Refer to
 * the section
 * \ref memory_overloading "Detailed Description"
 * for more information. In addition you need the configuration flag
 * --enable-freetrace to enable tracing of all free invocations. Pay
 * attention to the face that this is purely for debugging since memory
 * is actually never freed when enabling this.
 *
 * \param fp file pointer where logging information shall be written to
 * \return 0 in case of success otherwise negative error number.
 */
int memtrace_print_free_log( FILE* fp );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef MEMTRACE_H */
