/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef REFCNT_H
#define REFCNT_H

#include <stdio.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file refcnt.h
    \brief memory management by reference counting
    \defgroup refcnt_api Reference Counting
    \ingroup memory_api
    @{

    Reference  counting   is  common  technique  within   object  oriented
    programming and  often used when  garbage collection is not  an option
    e.g. due  to non  deterministic run time  characteristics, performance
    penalties, etc.

    A good introduction to this technique can be found within the
    <a href="https://developer.apple.com/library/mac/documentation/Cocoa/Conceptual/MemoryMgmt/Articles/MemoryMgmt.html#//apple_ref/doc/uid/10000011i">Advanced Memory Management Programming Guide</a>
    provided by Apple Inc.

    The  following  methods  provide  a general  interface  for  reference
    counting. The main  reason for using them is to  clearly indicate this
    design pattern within your code.  Currently this is solely used within
    cutil's string data type string_t. */


/*!
 * reference  counter  object,  currently   providing  solely  the  count
 * element. Make sure to integrate this  as first element within the data
 * structure which shall be managed by ref counting. */
typedef struct {
  int refcnt; /*!< reference counter */
} t_ref;


/*!
 * create new referenced object
 *
 * Allocate memory for a new referenced object and initialize
 * its reference counter to one.
 *
 * \param size size of the memory to be allocated
 * \return pointer to object of type size
 */
t_ref* ref_new( const int size );


/*!
 * retrain referenced object
 *
 * Retains given referenced object by incrementing its reference counter.
 * This requires  that the first element  within compound object r  is of
 * type t_ref.
 *
 * \param r pointer to referenced object, its first element must
 *          be of type t_ref
 */
void ref_retain( void* r );


/*!
 * release reference object
 *
 * Releases  given  referenced  object   by  decrementing  its  reference
 * counter. It the counter value hits zero the memory is reserved for the
 * object deallocated by invoking cul_free().
 *
 * \param r pointer to referenced object, its first element must
 *          be of type t_ref
 */
void ref_release( void* r );


/*! @} */

#ifdef __cplusplus
extern "C" {
#endif

#endif /* #ifndef REFCNT_H */

