/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef REFSTRING_H
#define REFSTRING_H

#include <olcutils/refcnt.h>
#include <olcutils/slist.h>
#include <olcutils/hashfunc.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file refstring.h
    \brief length delimited strings

    \addtogroup refstring_api References to Length Delimited Character Strings
    \ingroup functional_programming_api
    @{

     API  functions to  create  and manipulate  references  to length  delimited
     character strings. Instead  of a copy operation the used  data type referes
     to a  substring of a given  references which makes parsing  operations much
     more efficient. For more detailed explanation about the motivation refer to
     section \ref string_datatyp "Length Delimited String Datatyp".

     The most important API function are as follows:

     - string_new_from()   creates a new referenced string object from a C-string
     - string_retain()     retains a flat copy to string reference object, inc refcount
     - string_retain_sub() retains a flat copy to substring of the reference object
     - string_release()    releases a retained reference, decremnt refcount
     - string_new_clone()  duplicates a referenced string object
     - string_tmp_cstring_from() copy string to a (tempory) c-string buffer
 */


struct s_lstring;

/*! holds references to length delimited character strings */
typedef struct {
  t_ref ref; /*!< recount */
  struct s_lstring* p_lstring; /*!< internal stirng representation */
  int size;  /*!< size of managed bytes within referenced data content */
  char* p;   /*!< pointer to reference data content */
} string_t;


/*!
 * creates a new empty reference string object of given size
 *
 * \param  size size of managed string object in bytes
 * \return pointer pointer to newly created refstring object or
 *         NULL when out of memory
 */
string_t* string_new( const int size );


/*!
 * creates a new reference string object by copying NULL terminated C-String
 *
 * \param p_cstr NULL terminated C-String
 * \return pointer pointer to newly created refstring object or
 *         NULL when out of memory
 */
string_t* string_new_from( const char* p_cstr );


/*!
 * creates a new reference to a string reprentation
 *
 * The number  of objects which are  refering the given string  is stored
 * within  an internal  reference counter.  Whenever the  string used  by
 * another entity  the ref count  value needs be increased.  The function
 * string_retain() creates a new string_t  instance and initializes it to
 * refer to  the reference  data input  value s  whose refcount  value is
 * increased.
 *
 * \param s string object to be retained
 * \return new reference to string represented by s
 */
string_t* string_retain( string_t* s );


/*!
 * creates a new reference to a substring of the given string reference
 *
 * Similar to the function string_retain but instead of refering
 * the complete string, a substring of the given range is used.
 *
 * \param i string object to be retained
 * \param begin begin of the substring's range, count start from zero
 * \param end index of last character of the substring's range starting from zero
 * \return new reference to string represented by s
 */
string_t* string_retain_sub( string_t* i, int begin, int end );


/*!
 * release an external reference to the given string
 *
 * Decrements reference counting  for the given string.  If it reaches
 * zero frees all memory managed by given string representation.
 *
 * \param s string object to be released
 */
void string_release( string_t* s );


/*!
 * delivers a deep copy of the given string
 *
 * \param i string to copy
 * \return pointer to new string object or NULL in case of error
 */
string_t* string_new_clone( string_t* i );


/*!
 * print cutils's strings to standard output
 *
 * \param s string to be printed
 */
void string_print( const string_t* s );


/*!
 * print cutils's strings to buffered file descriptor
 *
 * \param fp pointer to stdlib FILE structure for output
 * \param s string to be printed
 */
void string_fprint( FILE* fp, const string_t* s );


/*!
 * print cutils's strings to standard output with carriage return
 *
 * \param s string to be printed
 */
void string_println( const string_t* s );


/*!
 * copy cutils's string buffer to external buffer
 *
 * This  serves to  tranform cutil's  string representation  to the  NULL
 * terminated format the c-library uses. The copied string is cut in case
 * it does not fit into the output buffer.
 *
 * \param s string to be copied from
 * \param c_string destination to copy to
 * \param c_string_size sizeo of the buffer c_string destination
 * \return pointer to c_string
 */
const char* string_tmp_cstring_from( const string_t* s, char* c_string, const int c_string_size );


/*!
 * cut of preceding blank characters in given string
 *
 * The function removes all  preceding blank characters (carriage return,
 * line feed, tabulator  and spaces) in given string.  The internal state
 * of  the referenced  string is  directly  altered. No  new objects  are
 * created.
 *
 * \param s string to be trimmed
 * \return input pointer
 */
string_t* string_triml( string_t* s );


/*!
 * cut of trailing blank characters in given string
 *
 * The function  removes all trailing blank  characters (carriage return,
 * line feed, tabulator  and spaces) in given string.  The internal state
 * of  the referenced  string is  directly  altered. No  new objects  are
 * created.
 *
 * \param s string to be trimmed
 * \return input pointer
 */
string_t* string_trimr( string_t* s );


/*!
 * cut of preceeding and trailing blank characters in given string
 *
 * The function  removes both,  preceeding and trailing  blank characters
 * (carriage return,  line feed, tabulator  and spaces) in  given string.
 * The internal  state of the  referenced string is directly  altered. No
 * new objects are created.
 *
 * \param s string to be trimmed
 * \return input pointer
 */
string_t* string_trim( string_t* s );


/*!
 * calculate hash value for cutils string object
 *
 * Refer to function hf_bitwisehash() for more information about the used
 * algorithm.
 *
 * \param s string object to calculate the hash value for
 * \return unique hash value corresponding to character buffer
 *         managed by s
 */
uint32_t string_hash( const string_t* s );


/*!
 * split cutil's string separated by delimeter characters
 *
 * The function splits a given cutil's string separated by the characters
 * specified by the  given NULL terminated delimeter string  and puts the
 * result to the  output list. The number of elements  to be generated at
 * maximum  can be  limited.  The  last list  element  provides the  rest
 * of  string in  case  the  the number  of  split  elements exceeds  the
 * limitation.
 *
 * \param s string to be splitted up
 * \param delims NULL terminated array of delimiter characters
 * \param limit maximum amount of generated list entries
 * \return list with elements of splitted strings
 */
slist_t* string_split( string_t* s, const char* delims, int limit );


/*!
 * split cutil's string into list of lines
 *
 * The  function splits  a given  cutil's  string separated  by the  line
 * separator characters (combination of CR  and/or LF) puts the result to
 * the output list.
 *
 * \param s string to be splitted up
 * \return list with elements of splitted strings
 */
slist_t* string_split_lines( string_t* s );


/*!
 * s_strlcpy -- size-bounded string copying and concatenation
 *
 * for more detailed information refer to: bsd man page strlcpy
 *
 * taken from:
 *   http://stackoverflow.com/questions/2933725/my-version-of-strlcpy
 */
size_t cstr_strlcpy(char *dest, const char *src, size_t len);


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef REFSTRING_H */
