/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SLIST_H
#define SLIST_H

#include <olcutils/lambda.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file slist.h
    \brief slingly linked lists

    \addtogroup slist_api Singly Linked Lists
    \ingroup functional_programming_api
    @{

     Concatenation of cons cells

     Empty elements (lists) are explicitly allowed. Refer to http://en.wikipedia.org/wiki/Cons
     for more detailed explanation about the specific meaning.

     ~~~
     [ e1  p ] [ e2  p ] [e3  nil]
          |      ^  |     ^
          -------|  |-----|

     [ e1  p ] [ e2  p ] [e3  p]  [ nil  nil ]
          |      ^  |     ^   |      ^
          -------|  |-----|   ------|
     ~~~

     - slist_alloc()   creates an empty list
     - slist_first()   returns first element
     - slist_rest()    returns new list without first element
     - slist_prepend() pushes elements in front of an (possibly empty)
                       list non destructively
     - slist_append()  appends at the end of the list (destructive operation)
     - slist_map()     creates a new list by mapping a function to each element
     - slist_reduce()  applys operator to first element and the rest of the list
                       recursively
     - slist_doseq()   maps void function foreach element
 */


/*! slingly link list data type */
typedef struct slist_struct
{
  void* data; /*!< void pointer to list element data */
  struct slist_struct *next; /*!< pointer to next element or NULL */
  int count; /*!< number of elements */
} slist_t;


/*!
 * creates a new empty singly linked list
 *
 * creates a new singly linked list without any elements in it.
 * \return pointer to (empty) list head
 */
slist_t* slist_alloc( void );


/*!
 * frees first element in list
 *
 * The first element or the so called head of the list
 * and returns a pointer to the next list element or
 * NULL if a next element does not exist.
 *
 * \param l pointer to list element to be freed
 * \return pointer to next list element or NULL if non existing
 */
slist_t* slist_free_first( slist_t* l );


/*!
 * frees all list element pointers
 *
 * iterates through all list elements and frees pointers
 * This function does not free the data itself! Use
 * slist_free_deep() when those shall be also freed.
 *
 * \param l list (pointers) to be freed
 */
void slist_free( slist_t* l );


/*!
 * frees all list elements
 *
 * frees all list elements with element data.
 *
 * \param l list to be freed.
 * \param free_op function pointer to element free operation
 */
void slist_free_deep( slist_t* l, lambda_t free_op );

/*!
 * appends element to the end of the list
 *
 * Append element to the end of the given list. Pay attention
 * to the fact that slist_append() needs to iterate through
 * all list elements so use instead the function slist_prepend()
 * whenever possible.
 *
 * \param l list to append element to
 * \param data element data pointer of the appended list element
 * \return pointer to the given list
 */
slist_t* slist_append( slist_t* l, void* data );


/*!
 * Prepends element to the head of the list
 *
 * Append element to the head of the given list.
 *
 * \param l list to prepend element to
 * \param data element data pointer of the prepended list element
 * \return pointer to the new list head
 */
slist_t* slist_prepend( slist_t* l, void* data );


/*!
 * Return the first element or head of the given list
 *
 * \param l list
 * \return element data of the first element or head of the given list
 */
void* slist_first( slist_t* l );


/*!
 * Return the "rest" of a list
 *
 * As in Lisp lists (or con cells) are considered to be consisted out
 * of the first element and a list of all remaining elements except
 * the first.
 *
 * \param l list
 * \return pointer to the rest of the given list or NULL if empty
 */
slist_t* slist_rest( slist_t* l );


/*!
 * Return the number of elements within a list
 *
 * This is a O(1) operation as in Clojure!
 *
 * \param l list
 * \return number of elements within list
 */
int slist_cnt( slist_t *l );

/*!
 * Return logical true when list is empty
 *
 * \param l list
 * \return 1 when list is empty, otherwise 0
 */
int slist_empty( slist_t* l );


/*!
 * Maps lambda operation to given list
 *
 * Creates a new output list where each element is the result of a lambda
 * function mapped to the given input list. In case of memory allocation error
 * the function ofm_free_op() is invoked for each newly created elements
 * in case it is provided. This allows to take care for memory leaks.
 *
 * \param l input list
 * \param f function which is mapped to each element of l
 * \param ofm_free_op pointer to free operation for newly created elements or NULL
 * \return pointer to new list
 */
slist_t* slist_map( slist_t* l, lambda_t f, lambda_t ofm_free_op );


/*!
 * Invokes lambda function for each list element
 *
 * The given lambda function is invoked for each list element.
 *
 * \param l list
 * \param f function to be invoked with each element
 */
void slist_doseq( slist_t* l, lambda_t f );


/*!
 * Reduce list with lambda function
 *
 * Starts iteration with the first to elements of l and applies
 * lambda function f to them. Then repeatedly invoke lambda function
 * with result of previous invocation and next list element until
 * the last element is reached. If the free operation ofm_free_op
 * is specified (!= NULL) its is invoked for the temporarily created
 * results of lambda invocations.
 *
 * \param l list to be reduced
 * \param f lambda function
 * \param ofm_free_op free operation for temporary results of f or NULL
 * \return result of last invocation of f
 */
void* slist_reduce( slist_t* l, lambda2_t f, lambda_t ofm_free_op );


/*!
 * Reverse a given list
 *
 * Creates a new list in reverse order to a given list. Note
 * that the elements are not copied. The data element pointers
 * remain the same.
 *
 * \param l list
 * \return pointer to newly created list in reverse order to l
 */
slist_t* slist_reverse( slist_t* l );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SLIST_H */
