/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <error.h>
#include <olcutils/alloc.h>
#include <config.h>

#ifndef ENABLE_LIFETIME_ALLOC_TRACE

static cul_allocstat_t allocstat;
static int is_initialized = 0;
static pthread_mutex_t   alloctrace_mutex = PTHREAD_MUTEX_INITIALIZER; /*!< mutex for access protection */


#ifdef ENABLE_ALLOC_TRACE

#define ADDR_TRACE_TABLE_SIZE  20000L

static void init_alloc_trace( cul_alloc_trace_t* p )
{
  if( p->addr_table_size == 0L ) {
    p->addr_table_size = ADDR_TRACE_TABLE_SIZE;
    p->addr_table_head = 0L;
    p->addr_table = malloc( p->addr_table_size * sizeof(cul_addr_dbg_info_t) );
    assert( pthread_mutex_init( & p->mutex, 0 ) == 0 );
    assert( p->addr_table != NULL );
  }
}

static void alloc_trace_insert( cul_alloc_trace_t* p, void* addr, const int size,
                                const char* func, const char* file, const int line )
{
  cul_addr_dbg_info_t* p_kv;

  pthread_mutex_lock( & p->mutex );
  assert( p->addr_table_head < p->addr_table_size );
  p_kv = p->addr_table + p->addr_table_head++;
  p_kv->addr = addr;
  p_kv->size = size;
  p_kv->dbg.func = func;
  p_kv->dbg.file = file;
  p_kv->dbg.line = line;
  pthread_mutex_unlock( & p->mutex );
}

static long alloc_trace_find_ptr( const cul_alloc_trace_t* p, const void* addr )
{
  long result = -1, i;

  for( i=0L; i < p->addr_table_head; ++i ) {
    if( p->addr_table[i].addr == addr ) {
      result = i;
      break;
    }
  }

  return result;
}

static int alloc_trace_remove( cul_alloc_trace_t* p, const void* addr )
{
  long i;

  pthread_mutex_lock( & p->mutex );
  i = alloc_trace_find_ptr( p, addr );
  if( i >= 0 ) {
    long later_elements = p->addr_table_head - i;
    if( later_elements > 0 ) {
    /* found and more than one element present, so move remaining */
      memmove( p->addr_table + i, p->addr_table + i + 1 ,
               later_elements * sizeof(cul_addr_dbg_info_t) );
    }
    --p->addr_table_head;
  }
  pthread_mutex_unlock( & p->mutex );

  if( i < 0 )
    return -1;
  else
    return 0;
}

#else


#pragma GCC diagnostic ignored "-Wunused-function"

static void init_alloc_trace( cul_alloc_trace_t* p_alloc_trace ) {}
static void alloc_trace_insert( cul_alloc_trace_t* p, void* addr, const size_t size,
                                const char* func, const char* file, const int line ) {}
static long alloc_trace_find_ptr( const cul_alloc_trace_t* p, const void* addr ) { return -1; }

#pragma GCC diagnostic pop


#endif /* #ifdef ENABLE_ALLOC_TRACE */

#ifdef ENABLE_OOB_CHECK
static const char alloc_prefix[]  = { 0xFF, 0xFE, 0xFD, 0xFC };
static const char alloc_postfix[] = { 0xEB, 0xEC, 0xED, 0xEE };

int cul_check_for_oob_violation( void* _ptr )
{
  void* ptr = _ptr - sizeof(size_t) - sizeof(alloc_prefix);
  size_t size = *((size_t *)ptr);
  long trace_idx = -1L;
  char locstr[80] = { '\0' };
  int flags = 0;

  if( memcmp( alloc_prefix, _ptr-sizeof(alloc_prefix), sizeof(alloc_prefix) ) ) {
    strncat( locstr, "prefix", 10 );
    flags |= CUL_PREFIX_VIOLATED_FLAG;
  }

  if( memcmp( alloc_postfix, _ptr+size, sizeof(alloc_postfix) ) ) {
    strncat( locstr, ", postfix", 10 );
    flags |= CUL_POSTFIX_VIOLATED_FLAG;
  }

  if( flags ) {
    fprintf( stderr, "%s, %d: memory out of bounds access violation discovered in %s ",
             __func__, __LINE__, locstr );

    trace_idx = alloc_trace_find_ptr( & allocstat.alloc_trace, _ptr );
    if( trace_idx >= 0 ) {
      fprintf( stderr, "at addr: %p, size: %ld, func: %s, file: %s, line: %d\n",
               allocstat.alloc_trace.addr_table[trace_idx].addr,
               (unsigned long) ( allocstat.alloc_trace.addr_table[trace_idx].size ),
               allocstat.alloc_trace.addr_table[trace_idx].dbg.func,
               allocstat.alloc_trace.addr_table[trace_idx].dbg.file,
               allocstat.alloc_trace.addr_table[trace_idx].dbg.line );
    } else {
      fprintf( stderr, "at unknown allocator!\n" );
    }

#ifdef ENABLE_OOB_ASSERT
    fprintf(stderr, "memory out of bounds access violation discovered\n" );
    exit(-1);
#endif
  }

  return flags;
}


#else


#pragma GCC diagnostic ignored "-Wunused-const-variable"

static const char alloc_prefix[0];
static const char alloc_postfix[0];

int cul_check_for_oob_violation( void* _ptr )
{
  return 0;
}

#pragma GCC diagnostic pop


#endif /* #ifdef ENABLE_OOB_CHECK */


static void initialize( void )
{
  if( is_initialized )
    return;

  memset( & allocstat, 0, sizeof( cul_allocstat_t ) );
  init_alloc_trace( & allocstat.alloc_trace );

  is_initialized = 1;
}


#ifdef ENABLE_ALLOC_TRACE

void* _cul_malloc( size_t size, const char* func, const char* file, const int line, const int ext )
{
  void* p;

  initialize();

  p = malloc( size + sizeof(size_t) + sizeof(alloc_prefix) + sizeof(alloc_postfix) );
  if( p )
  {
    *((size_t *)p) = size;
    p += sizeof(size_t);

    pthread_mutex_lock( & alloctrace_mutex );
    ++allocstat.nr_allocs;
    allocstat.mem_allocated += size;
    if( allocstat.mem_allocated > allocstat.max_allocated )
      allocstat.max_allocated = allocstat.mem_allocated;

#ifdef ENABLE_OOB_CHECK
    memcpy( p, alloc_prefix, sizeof(alloc_prefix) );
    p += sizeof(alloc_prefix);
    memcpy( p + size, alloc_postfix, sizeof(alloc_postfix) );
#endif /* #ifdef ENABLE_OOB_CHECK */

    alloc_trace_insert( & allocstat.alloc_trace, p, size, func, file, line );
    pthread_mutex_unlock( & alloctrace_mutex );
  }
  else
  {
    pthread_mutex_lock( & alloctrace_mutex );
    ++allocstat.nr_allocs_failed;
    pthread_mutex_unlock( & alloctrace_mutex );
  }

  return p;
}

void* _cul_realloc( void* _ptr, size_t size, const char* func, const char* file, const int line, const int ext )
{
  void* ptr = _ptr;
  void* p;
  size_t prev_size = 0;
  int alloced = 1;

  initialize();

  if( ptr )
  {
#ifdef ENABLE_OOB_CHECK
    cul_check_for_oob_violation( ptr );
#endif /* #ifdef ENABLE_OOB_CHECK */

    ptr -= sizeof(size_t) + sizeof(alloc_prefix);
    prev_size = *((size_t *)ptr);
    alloced = 0;
  }

  p = realloc( ptr, size + sizeof(size_t) + sizeof(alloc_prefix) + sizeof(alloc_postfix) );
  if( p )
  {
    *((size_t *)p) = size;
    p += sizeof(size_t);

    pthread_mutex_lock( & alloctrace_mutex );
    allocstat.nr_allocs += alloced;
    allocstat.mem_allocated += size - prev_size;
    if( allocstat.mem_allocated > allocstat.max_allocated )
      allocstat.max_allocated = allocstat.mem_allocated;

#ifdef ENABLE_OOB_CHECK
    memcpy( p, alloc_prefix, sizeof(alloc_prefix) );
    p += sizeof(alloc_prefix);
    memcpy( p + size, alloc_postfix, sizeof(alloc_postfix) );
#endif /* #ifdef ENABLE_OOB_CHECK */
    pthread_mutex_unlock( & alloctrace_mutex );

#ifdef ENABLE_ALLOC_TRACE
    if( _ptr ) {
      if( alloc_trace_remove( & allocstat.alloc_trace, _ptr ) ) {
        fprintf( stderr, "____ double free condition in file: %s, func %s, line %d error!\n", file, func, line );
        assert( 0 );
      }
    }
    alloc_trace_insert( & allocstat.alloc_trace, p, size, func, file, line );
#endif /* #ifdef ENABLE_ALLOC_TRACE */
  }
  else
  {
    pthread_mutex_lock( & alloctrace_mutex );
    ++allocstat.nr_allocs_failed;
    pthread_mutex_unlock( & alloctrace_mutex );
  }

  return p;
}

void _cul_free( void* _ptr, const char* func, const char* file, const int line, const int ext )
{
  void* ptr;
  size_t size;

  initialize();

  ptr = _ptr - sizeof(size_t) - sizeof(alloc_prefix);
  size = *((size_t *)ptr);

  pthread_mutex_lock( & alloctrace_mutex );
  --allocstat.nr_allocs;
  allocstat.mem_allocated -= size;

#ifdef ENABLE_OOB_CHECK
  cul_check_for_oob_violation( _ptr );
#endif /* #ifdef ENABLE_OOB_CHECK */

#ifdef ENABLE_ALLOC_TRACE
  if( alloc_trace_remove( & allocstat.alloc_trace, _ptr ) ) {
    fprintf( stderr, "____ double free condition in file: %s, func %s, line %d error!\n", file, func, line );
    assert( 0 );
  }
#endif /* #ifdef ENABLE_ALLOC_TRACE */
  pthread_mutex_unlock( & alloctrace_mutex );


  free( ptr );
}

#else

void* _cul_malloc( size_t size, const char* func, const char* file, const int line, const int ext ) {
  return malloc( size );
}

void* _cul_realloc( void* _ptr, size_t size, const char* func, const char* file, const int line, const int ext ) {
  return realloc( _ptr, size );
}

void _cul_free( void* _ptr, const char* func, const char* file, const int line, const int ext ) {
  free( _ptr );
}

#endif /* #ifdef ENABLE_ALLOC_TRACE */

cul_allocstat_t get_allocstat( void )
{
  cul_allocstat_t s;
  initialize();

  pthread_mutex_lock( & alloctrace_mutex );
  s = allocstat;
  pthread_mutex_unlock( & alloctrace_mutex );

  return s;
}

#endif /* #ifndef ENABLE_LIFETIME_ALLOC_TRACE */
