/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <olcutils/hashmap.h>
#include <syslog.h>
#include <unistd.h>
#include <execinfo.h>
#include <olcutils/alloc.h>
#include <config.h>

#ifdef ENABLE_LIFETIME_ALLOC_TRACE

/*
 * Debug instrumentation for memory references to be externally freed
 *
 * 2023@valeo/ol
 */

/* --- private implementation --- */

/* 60 minutes */
long g_auto_write_trace_table_once_after_seconds = 3600;

/* 200 elements are sufficient for print out since we write also to a file */
#define ADDR_TRACE_TABLE_SIZE  200


extern char *program_invocation_name;
extern char *program_invocation_short_name;


static cul_allocstat_t   allocstat;                 /*!< allocation statistics */
static int               alloctrace_initialized = 0;  /*!< singleton flag */
static pthread_mutex_t   alloctrace_mutex = PTHREAD_MUTEX_INITIALIZER; /*!< mutex for access protection */
static hm_t*             alloctrace_hm = NULL;        /*!< alloctrace hash map */


/*!
 * provides information from where a message has been requested
 */
typedef struct {
  void*             ptr;        /*!< pointer to observed address space as returned by malloc (64 bit on host platform) */
  size_t            size;       /*!< size of memory block */
  int               type;       /*!< type of memory block */
  char*             file;       /*!< pointer to requesting file name */
  int               line;       /*!< line number */
  char*             func;       /*!< pointer to function or member name which allocated */
  long              ts_ms;      /*!< timestampe in milliseconds */
} alloctrace_info_block;


static long get_uptime_ms( void )
{
  static int initialized = 0;
  static struct timespec first_invocation;
  struct timespec current_time;
  long diff_ms;
  int error = 0;

  if( ! initialized ) {
    /* make sure that last_write_op is always initialized */
    error = clock_gettime( CLOCK_MONOTONIC_RAW, &first_invocation );
    if( error ) {
      fprintf( stderr, "%s,%d: could not get initial timestamp error!\n",
              __func__, __LINE__  );
      first_invocation.tv_sec = 0;
      first_invocation.tv_nsec = 0L;
    }

    initialized = 1;
  }

  if( ! error ) {
    error = clock_gettime( CLOCK_MONOTONIC_RAW, &current_time );
    if( error ) {
      fprintf( stderr, "%s,%d: could not get current timestamp error!\n",
              __func__, __LINE__  );
    }
  }

  if( ! error ) {
    diff_ms = 1000 * (current_time.tv_sec - first_invocation.tv_sec) +
      (current_time.tv_nsec - first_invocation.tv_nsec) / 1000000L;
    return diff_ms;
  } else {
    return (long)error;
  }
}


static void release_alloctrace_info_block( alloctrace_info_block* p )
{
  if( p ) {
    free( p );
  }
}


static void* free_element_handler( void* p )
{
  release_alloctrace_info_block( p );
  return NULL;
}


static alloctrace_info_block* init_alloctrace_info_block(
  const void* ptr,
  const size_t size,
  const int type,
  const char* file,
  const int line,
  const char* func )
{
  alloctrace_info_block* p;

  p = malloc( sizeof( alloctrace_info_block ) );
  if( p == NULL ) {
    fprintf( stderr, "%s,%d: out of memory error!\n", __func__, __LINE__  );
    return p;
  }

  p->file = (char *)file;
  p->func = (char *)func;
  p->line = line;
  p->size = size;
  p->type = type;
  p->ts_ms = get_uptime_ms();

  return p;
}


static int alloctrace_init( void )
{
  FILE* fp;
  char alloctrace_control_file[80];
  char control_string[80];
  long control_val;

  cul_alloc_trace_t *p_trace = &allocstat.alloc_trace;

  if( alloctrace_initialized ) {
    fprintf( stderr, "%s,%d: alloctrace already initialized!\n", __func__, __LINE__  );
    return -1;
  }

  alloctrace_hm = hm_alloc();
  if( alloctrace_hm == NULL ) {
    fprintf( stderr, "%s,%d: trace hash map could not be initialized!\n",
            __func__, __LINE__  );
    return -1;
  }

  memset( & allocstat, 0, sizeof( cul_allocstat_t ) );

  p_trace->addr_table_size = ADDR_TRACE_TABLE_SIZE;
  p_trace->addr_table_head = 0L;
  p_trace->addr_table = malloc( p_trace->addr_table_size * sizeof(cul_addr_dbg_info_t) );
  if( p_trace->addr_table == NULL ) {
    fprintf( stderr, "%s,%d: allocstat table could not be initialized error!\n",
            __func__, __LINE__  );
    hm_free_deep( alloctrace_hm, 0, free_element_handler );
    return -1;
  }

  snprintf( alloctrace_control_file, sizeof(alloctrace_control_file),
            "/tmp/%s_alloctrace_control", program_invocation_short_name );
  fp = fopen( alloctrace_control_file, "r" );
  if( fp ) {
    if( fread( control_string, sizeof(char), sizeof(control_string), fp ) > 0 ) {
      control_val = atol( control_string );
      if( control_val > 10 && control_val < 3600 * 24 * 30 ) {
        g_auto_write_trace_table_once_after_seconds = control_val;
        printf("%s,%d: auto write trace table after %ld seconds\n",
               __func__, __LINE__, g_auto_write_trace_table_once_after_seconds );
      } else {
        fprintf( stderr, "%s,%d: could not parse crtl file: %s error!\n",
                 __func__, __LINE__, alloctrace_control_file );
      }
    }
  } else {
    fprintf( stderr, "%s,%d: crtl file: %s not defined, auto write trace after %ld s!\n",
             __func__, __LINE__,
             alloctrace_control_file, g_auto_write_trace_table_once_after_seconds );
  }

  alloctrace_initialized = 1;

  return 0;
}

#if(0)
/* unused */
static int alloctrace_release( void )
{
  cul_alloc_trace_t* p_trace;

  if( ! alloctrace_initialized ) {
    fprintf( stderr, "%s,%d: alloctrace not initialized and requested to be freed!\n",
            __func__, __LINE__  );
    return -1;
  }
  alloctrace_initialized = 0;

  hm_free_deep( alloctrace_hm, 0, free_element_handler );

  p_trace = &allocstat.alloc_trace;
  p_trace->addr_table_head = 0;

  return 0;
}
#endif


static void alloctrace_write_table_to_progname_file( void );


static int alloctrace_add_ptr( const void* ptr, const size_t size, const int type, const char* file, const int line, const char* func )
{
  alloctrace_info_block* p_info_block;
  int error = 0;
  static int trace_table_written = 0;

  if( ! error ) {
    p_info_block = init_alloctrace_info_block( ptr, size, type, file, line, func );
    if( p_info_block == NULL ) {
      error = -1;
    }
  }

  if( ! error ) {
#pragma GCC diagnostic ignored "-Wpointer-to-int-cast"
    /* We have to  down cast the pointer  to be used as hash  key because the
       'ideal hash tries' implementation in cutils is 32 bit only. On  64 bit
       systems this typically creates a  compilation warning. But this can be
       safely ignored  because 32 bit  covers still  an address range  of 4GB
       which is  exceeding most  address ranges  of typical  applications and
       particulary the available memory of embedded devices. */
    hm_assoc( alloctrace_hm, 0, (hm_hash_t)ptr, p_info_block );
#pragma GCC diagnostic pop
  }

  if( ! trace_table_written && get_uptime_ms() > g_auto_write_trace_table_once_after_seconds * 1000L  ) {
    trace_table_written = 1;

    alloctrace_write_table_to_progname_file();
  }

  return error;
}

static alloctrace_info_block* alloctrace_find_ptr( const void* ptr )
{
  hm_leaf_node_t* found;
  alloctrace_info_block* p_info_block = NULL;

#pragma GCC diagnostic ignored "-Wpointer-to-int-cast"
  found = hm_find( alloctrace_hm, (hm_hash_t)ptr );
#pragma GCC diagnostic pop

  if( found == NULL ) {
    fprintf( stderr, "%s,%d: alloctrace element %p could not be removed error!\n",
             __func__, __LINE__, ptr  );
  } else {
    p_info_block = found->val;
  }

  return p_info_block;
}

static int alloctrace_remove_ptr( const void* ptr )
{
  alloctrace_info_block* p_info_block;
  int error = 0;

  p_info_block = alloctrace_find_ptr( ptr );

  if( p_info_block ) {
    release_alloctrace_info_block( p_info_block );

#pragma GCC diagnostic ignored "-Wpointer-to-int-cast"
    hm_dissoc( alloctrace_hm, 0, (hm_hash_t)ptr );
#pragma GCC diagnostic pop
  } else {
      fprintf( stderr, "%s,%d: alloctrace element %p could not be removed error!\n",
              __func__, __LINE__, ptr  );
      error = -1;
  }

  return error;
}


static FILE* alloctrace_write_table_fp = NULL;

static void* lambda_print_key_info_block( void* p_key, void* val )
{
  alloctrace_info_block* p_info_block = val;

  if( alloctrace_write_table_fp ) {
    fprintf( alloctrace_write_table_fp, "%ld %lu, %d, %lx %s %d %s\n",
             p_info_block->ts_ms,
             (unsigned long)(p_info_block->size),
             p_info_block->type,
             (unsigned long) *((uint32_t *)p_key),
             p_info_block->file,
             p_info_block->line,
             p_info_block->func );
  }

  return NULL;
}

static void alloctrace_write_table( const char* output_file_name )
{
  int error = 0;

  alloctrace_write_table_fp = fopen( output_file_name, "wt" );
  if( alloctrace_write_table_fp == NULL ) {
    fprintf( stderr, "%s,%d: Could not open alloctrace output file %s error!\n",
            __func__, __LINE__ , output_file_name );
    error = -1;
  }

  if( ! error ) {
    printf("%s,%d: write allocation trace table to %s\n",
           __func__, __LINE__, output_file_name );

    fprintf( alloctrace_write_table_fp, "ts[ms] size type pointer file line func\n" );
    hm_doseq( alloctrace_hm, 0, lambda_print_key_info_block );
    fclose( alloctrace_write_table_fp );
  }
}


static void alloctrace_write_table_to_progname_file( void )
{
  char alloctrace_filename[80];

  snprintf( alloctrace_filename, sizeof(alloctrace_filename),
            "/tmp/%s_alloctrace", program_invocation_short_name );
  alloctrace_write_table( alloctrace_filename );
}


/*!
 * testing of fundamental hash table function (adding/removing)
 */
#if(0)
static int alloctrace_test_hash( void )
{
  char alloctrace_filename[80];
  const int nr_tests = 10;
  const int size = 10;
  char* p_tab[nr_tests];
  int i;

  printf( "%s,%d,%s: start alloctrace test and write results to file '%s'.\n",
          __FILE__, __LINE__, __func__,  alloctrace_filename );

  snprintf( alloctrace_filename, sizeof(alloctrace_filename), "/tmp/%s_alloctrace", program_invocation_short_name );

  if( access( alloctrace_filename, F_OK ) == 0 ) {
    /* make sure that the test function is only exeucted once */
    fprintf( stderr, "%s,%d,%s: alloctrace test file %s exists, exit!\n",
            __FILE__, __LINE__, __func__,  alloctrace_filename );
    return -1;
  }

  /* allocate and add 10 elements to trace table */
  for( i=0; i<nr_tests; ++i ) {
    p_tab[i] = malloc( size );
    if( p_tab[i] == NULL ) {
      fprintf( stderr, "%s,%d,%s: out of memory error!\n",
              __FILE__, __LINE__, __func__ );
      while( --i >= 0 ) {
        free( p_tab[i] );
        return -1;
      }
    }

    if( alloctrace_add_ptr( p_tab[i], size, 0, __FILE__, __LINE__, __func__ ) ) {
      fprintf( stderr, "%s,%d,%s: could not add element %d, msg: %p!\n",
              __FILE__, __LINE__, __func__, i, p_tab[i] );
    } else {
      printf( "%s,%d,%s: added  ptr: %lx\n",
              __FILE__, __LINE__, __func__,  (unsigned long)i );
    }

    sleep( 1 );
  }

  /* remove all odd elements */
  for( i=0; i<nr_tests; ++i ) {
    if( i%2 ) {
      if( alloctrace_remove_ptr( p_tab[i] ) ) {
        fprintf( stderr, "%s,%d,%s: could not remove element: %d, msg: %p!\n",
                __FILE__, __LINE__, __func__, i, p_tab[i] );
      }
    }
  }

  alloctrace_write_table( alloctrace_filename );

  (void)alloctrace_release();
  for( i=0; i<nr_tests; ++i ) {
    free( p_tab[i] );
  }

  return 0;
}
#endif

#if(0)
static int alloctrace_test_autowrite( void )
{
  const int nr_tests = 70;
  char* p_tab[nr_tests];
  const int size = 10;
  int i;

  /* allocate and add 10 elements to trace table */
  for( i=0; i<nr_tests; ++i ) {
    p_tab[i] = malloc( size );
    if( p_tab[i] == NULL ) {
      fprintf( stderr, "%s,%d,%s: out of memory error!\n",
              __FILE__, __LINE__, __func__ );
      while( --i >= 0 ) {
        free( p_tab[i] );
        return -1;
      }
    }

    if( alloctrace_add_ptr( p_tab[i], size, 0, __FILE__, __LINE__, __func__ ) ) {
      printf( "%s,%d,%s: could not add element %d, msg: %p!\n",
              __FILE__, __LINE__, __func__, i, p_tab[i] );
    } else {
      printf( "%s,%d,%s: added  ptr: %lx\n",
              __FILE__, __LINE__, __func__,  (unsigned long)i );
    }

    sleep( 1 );
  }

  (void)alloctrace_release();
  for( i=0; i<nr_tests; ++i ) {
    free( p_tab[i] );
  }

  return 0;
}
#endif

#if(0)
static int write_callback_trace( void )
{
  FILE* fp;
  char callback_trace_filename[256];
  const size_t max_size = 50;
  void *array[max_size];
  size_t size;
  char **strings;
  size_t i;

  static int trace_num = 0;

  snprintf( callback_trace_filename,
            sizeof(callback_trace_filename),
            "/tmp/%s_callback_trace_%d",
            program_invocation_short_name,
            trace_num );

  fp = fopen( callback_trace_filename, "wt" );
  if( fp == NULL ) {
    fprintf( stderr, "%s,%d,%s: could not open callback trace file %s error!\n",
            __FILE__, __LINE__, __func__, callback_trace_filename );
    return -1;
  }

  size = backtrace (array, max_size );
  strings = backtrace_symbols( array, size );

  fprintf( fp, "Obtained %zd stack frames.\n", size );

  for( i = 0; i < size; i++ )
    fprintf( stderr, "%s\n", strings[i] );

  ++trace_num;

  fclose( fp );
  free( strings );

  return 0;
}
#endif


#ifdef ENABLE_OOB_CHECK
static const char alloc_prefix[]  = { 0xFF, 0xFE, 0xFD, 0xFC };
static const char alloc_postfix[] = { 0xEB, 0xEC, 0xED, 0xEE };


int cul_check_for_oob_violation( void* _ptr )
{
  void* ptr = _ptr - sizeof(size_t) - sizeof(alloc_prefix);
  size_t size = *((size_t *)ptr);
  alloctrace_info_block* p_info_block;
  char locstr[80] = { '\0' };
  int flags = 0;

  if( memcmp( alloc_prefix, _ptr-sizeof(alloc_prefix), sizeof(alloc_prefix) ) ) {
    strncat( locstr, "prefix", 10 );
    flags |= CUL_PREFIX_VIOLATED_FLAG;
  }

  if( memcmp( alloc_postfix, _ptr+size, sizeof(alloc_postfix) ) ) {
    strncat( locstr, ", postfix", 10 );
    flags |= CUL_POSTFIX_VIOLATED_FLAG;
  }

  if( flags ) {
    fprintf( stderr, "%s, %d: memory out of bounds access violation discovered in %s ",
             __func__, __LINE__, locstr );

    p_info_block = alloctrace_find_ptr( _ptr );
    if( p_info_block ) {
      fprintf( stderr, "at addr: %p, size: %lu, func: %s, file: %s, line: %d\n",
               _ptr,
               (unsigned long)(p_info_block->size),
               p_info_block->func,
               p_info_block->file,
               p_info_block->line );
    } else {
      fprintf( stderr, "at unknown allocator!\n" );
    }

#ifdef ENABLE_OOB_ASSERT
    fprintf(stderr, "memory out of bounds access violation discovered\n" );
    exit(-1);
#endif
  }

  return flags;
}

#else
static const char alloc_prefix[0];
static const char alloc_postfix[0];

int cul_check_for_oob_violation( void* ptr )
{
  return 0;
}
#endif /* #ifdef ENABLE_OOB_CHECK */



/* --- public interface --- */

void* _cul_malloc( size_t size, const char* func, const char* file, const int line, const int ext )
{
  if( ext )
  {
    void* p = NULL;

    pthread_mutex_lock( & alloctrace_mutex );

    if( ! alloctrace_initialized ) {
      if( alloctrace_init() ) {
        pthread_mutex_unlock( & alloctrace_mutex );
        return NULL;
      }
    }

    p = malloc( size + sizeof(size_t) + sizeof( alloc_prefix ) + sizeof( alloc_postfix ) );

    if( p == NULL ) {
      fprintf( stderr, "%s,%d,%s: out of memory error!\n",
              file, line, func );

      ++allocstat.nr_allocs_failed;
    } else {
      *((size_t *)p) = size;
      p += sizeof(size_t);

      ++allocstat.nr_allocs;
      allocstat.mem_allocated += size;
      if( allocstat.mem_allocated > allocstat.max_allocated ) {
        allocstat.max_allocated = allocstat.mem_allocated;
      }

#ifdef ENABLE_OOB_CHECK
      memcpy( p, alloc_prefix, sizeof(alloc_prefix) );
      p += sizeof(alloc_prefix);
      memcpy( p + size, alloc_postfix, sizeof(alloc_postfix) );
#endif /* #ifdef ENABLE_OOB_CHECK */

      if( alloctrace_add_ptr( p, size, 1 /* type: ext */, file, line, func ) ) {
        fprintf( stderr, "%s,%d,%s: could not add ptr: %p!\n",
                 file, line, func, p );
      }
    }

    pthread_mutex_unlock( & alloctrace_mutex );

    return p;
  }
  else
  {
    return malloc( size );
  }
}

void* _cul_realloc( void* _ptr, size_t size, const char* func, const char* file, const int line, const int ext )
{
  if( ext )
  {
    void* ptr = _ptr;
    size_t prev_size = 0;
    int alloced = 1;

    pthread_mutex_lock( & alloctrace_mutex );

    if( ! alloctrace_initialized ) {
      if( alloctrace_init() ) {
        pthread_mutex_unlock( & alloctrace_mutex );
        return NULL;
      }
    }

    if( ptr )
    {
#ifdef ENABLE_OOB_CHECK
      cul_check_for_oob_violation( _ptr );
#endif /* #ifdef ENABLE_OOB_CHECK */
      ptr = _ptr - sizeof(size_t) - sizeof(alloc_prefix);
      prev_size = *((size_t *)ptr);
      alloced = 0;
    }

    if( _ptr ) {
      if( alloctrace_remove_ptr( _ptr ) ) {
        fprintf( stderr, "%s,%d,%s: could not remove ptr: %p!\n",
                 file, line, func, _ptr );
      }
    }

    ptr = realloc( ptr, size + sizeof( size_t ) + sizeof( alloc_prefix ) + sizeof( alloc_postfix ) );
    if( ptr == NULL ) {
      fprintf( stderr, "%s,%d,%s: out of memory error!\n",
               file, line, func );

      ++allocstat.nr_allocs_failed;
    } else {
      *((size_t *)ptr) = size;
      ptr += sizeof(size_t);

      allocstat.nr_allocs += alloced;
      allocstat.mem_allocated += size - prev_size;
      if( allocstat.mem_allocated > allocstat.max_allocated ) {
        allocstat.max_allocated = allocstat.mem_allocated;
      }

#ifdef ENABLE_OOB_CHECK
      memcpy( ptr, alloc_prefix, sizeof(alloc_prefix) );
      ptr += sizeof(alloc_prefix);
      memcpy( ptr + size, alloc_postfix, sizeof(alloc_postfix) );
#endif /* #ifdef ENABLE_OOB_CHECK */

      if( alloctrace_add_ptr( ptr, size, 1 /* type: ext */, file, line, func ) ) {
        fprintf( stderr, "%s,%d,%s: could not add ptrtr: %ptr!\n",
                 file, line, func, ptr );
      }
    }

    pthread_mutex_unlock( & alloctrace_mutex );

    return ptr;
  }
  else
  {
    return realloc( _ptr, size );;
  }

}

void _cul_free( void* _ptr, const char* func, const char* file, const int line, const int ext )
{
  if( ext )
  {
    void* ptr;
    size_t size;

    pthread_mutex_lock( & alloctrace_mutex );

    if( ! alloctrace_initialized ) {
      if( alloctrace_init() ) {
        pthread_mutex_unlock( & alloctrace_mutex );
        return;
      }
    }

    ptr = _ptr - sizeof(size_t) - sizeof(alloc_prefix);
    size = *((size_t *)ptr);

    --allocstat.nr_allocs;
    allocstat.mem_allocated -= size;

#ifdef ENABLE_OOB_CHECK
    cul_check_for_oob_violation( _ptr );
#endif /* #ifdef ENABLE_OOB_CHECK */

    if( alloctrace_remove_ptr( _ptr ) ) {
      fprintf( stderr, "%s,%d,%s: double free condition ptr: %p!\n",
               file, line, func, ptr );
    }

    free( ptr );

    pthread_mutex_unlock( & alloctrace_mutex );
  }
  else
  {
    free( _ptr );
  }
}

static void* lambda_print_insert_info_block( void* p_key, void* val )
{
  alloctrace_info_block* p_info_block = val;
  cul_alloc_trace_t *p_trace = &allocstat.alloc_trace;
  const int i = p_trace->addr_table_head;

  if( ! (p_trace->addr_table_head < p_trace->addr_table_size ) ) {
    return NULL;
  }

  p_trace->addr_table[i].addr = p_info_block->ptr;
  p_trace->addr_table[i].size = p_info_block->size;
  p_trace->addr_table[i].dbg.func = p_info_block->func;
  p_trace->addr_table[i].dbg.file = p_info_block->file;
  p_trace->addr_table[i].dbg.line = p_info_block->line;

  ++(p_trace->addr_table_head);
  return NULL;
}

static int fill_alloc_trace( void )
{
  cul_alloc_trace_t *p_trace = &allocstat.alloc_trace;
  p_trace->addr_table_head = 0;

  hm_doseq( alloctrace_hm, 0, lambda_print_insert_info_block );

  return 0;
}


cul_allocstat_t get_allocstat( void )
{
  pthread_mutex_lock( & alloctrace_mutex );

  if( alloctrace_initialized ) {
    alloctrace_write_table_to_progname_file();
    fill_alloc_trace();
  }

  pthread_mutex_unlock( & alloctrace_mutex );

  return allocstat;
}

#endif /* #ifdef ENABLE_LIFETIME_ALLOC_TRACE */
