/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <olcutils/hashmap.h>
#include <olcutils/alloc.h>

/*!
 * count bit population within an integer (32 bit)
 * counts number of least significant bits set to 1
 *
 * this is just for illustration and checking purposes
 */
int CTPop_check( hm_hash_t i )
{
  int j, c;

  c = 0;
  for(j=0; j<sizeof(i)*8; j++) {
    if(i & (1ul << j)) c++;
  }

  return c;
}


const static unsigned int SK5=0x55555555, SK3=0x33333333;
const static unsigned int SKF0=0xF0F0F0F; //, SKFF=0xFF00FF;

/*!
 * count bit population within an integer (32 bit)
 * counts number of least significant bits set to 1
 *
 * taken from "Ideal Hash Trees", by Phil Bagwell
 * http://lampwww.epfl.ch/papers/idealhashtrees.pdf
 */
int CTPop( hm_hash_t map )
{
  map-=((map>>1)&SK5);
  map=(map&SK3)+((map>>2)&SK3);
  map=(map&SKF0)+((map>>4)&SKF0);
  map+=map>>8;
  return (map+(map>>16))&0x3F;
}


static hm_hash_t hm_mask( hm_hash_t hash, int shift ) {
  return (hash >> shift) & 0x01f;
}

static hm_hash_t hm_bitpos( hm_hash_t hash, int shift ) {
  return ((hm_hash_t)1) << hm_mask(hash, shift);
}

static int hm_index(hm_hash_t bitmap, hm_hash_t bit) {
  return CTPop( bitmap & (bit - 1) );
}

hm_t* hm_alloc( void ) {
  hm_t* p;
  p = cul_malloc( sizeof(hm_t) );
  if( p )
    memset( p, 0, sizeof(hm_t) );

  return p;
}

hm_leaf_node_t* hm_find( hm_t* this, hm_hash_t hash )
{
  int bit = hm_bitpos( hash, this->shift );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;

  if((this->bitmap & bit) != 0)
  {
    p_inode = & this->nodes[hm_index(this->bitmap, bit)];

    switch( p_inode->typ)
    {
    case bitmap_index_node_type:
      p_lnode = hm_find( & p_inode->u.inode, hash );
      break;

    case leaf_node_type:
      p_lnode = &p_inode->u.lnode;
    }

    if( p_lnode && p_lnode->hash == hash )
      return p_lnode;
    else
      return NULL;
  }
  else
    return NULL;
}


hm_t* hm_assoc( hm_t* this, int shift, hm_hash_t hash, void* val )
{
  int bitpos = hm_bitpos( hash, shift );
  int idx = hm_index( this->bitmap, bitpos );;
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;
  hm_t* returned_hash = NULL;

  hm_hash_t collided_hash;
  void* collided_val;

  if( ! this )
    return NULL;

  if((this->bitmap & bitpos) != 0)
  {
    /* bitpos is used by either an index node or an leaf node  */
    if( this->nodes == NULL )
    {
      /* nodes have not been allocated yet. In that
         case initialize another index node structure */
      this->nodes = cul_malloc( sizeof(hm_inode_t) );
      if( this->nodes == NULL )
        return NULL;
      memset( this->nodes, 0, sizeof(hm_inode_t) );
      this->shift = shift;
      assert( idx == 0 );
    }

    p_inode = & this->nodes[idx];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      /* if index node append new node recursively there */
      returned_hash = hm_assoc( & p_inode->u.inode, shift+5, hash, val );
      break;

    case leaf_node_type:
      /* if leaf node node, check wether it carries the dame hash */
      p_lnode = &p_inode->u.lnode;
      if( p_lnode->hash == hash )
      {
        /* if so assign new value */
        p_lnode->val = val;
      }
      else
      {
        /* new and exiting hash differ, that means we have a collision, so create another index node */
        collided_hash = p_lnode->hash;
        collided_val = p_lnode->val;

        p_inode->typ = bitmap_index_node_type;
        p_inode->u.inode.bitmap = 0;
        p_inode->u.inode.nodes = NULL;

        returned_hash = hm_assoc( & p_inode->u.inode, shift+5, collided_hash, collided_val );
        if( returned_hash )
          returned_hash = hm_assoc( & p_inode->u.inode, shift+5, hash, val );
      }
    }
  }
  else
  {
    /* bitpos is not used, so insert new leaf node */
    hm_leaf_node_t nleaf = { hash, val };
    hm_inode_t* new_array = cul_realloc( this->nodes, sizeof(hm_inode_t) * (n+1) );
    if( new_array != NULL )
    {
      this->bitmap |= bitpos;
      this->shift = shift;
      this->nodes = new_array;
      memmove( & this->nodes[idx+1], & this->nodes[idx], sizeof( hm_inode_t ) * (n-idx) );

      this->nodes[idx].u.lnode = nleaf;
      this->nodes[idx].typ = leaf_node_type;
      returned_hash = this;
    }
  }

  return returned_hash;
}


hm_t* hm_dissoc( hm_t* this, int shift, hm_hash_t hash )
{
  int n = CTPop( this->bitmap );
  int bitpos = hm_bitpos( hash, shift );
  int idx = hm_index( this->bitmap, bitpos );;
  hm_inode_t* p_inode;
  hm_t* returned_hash = NULL;

  if( ! this )
    return NULL;

  if((this->bitmap & bitpos) != 0)
  {
    /* bitpos is used by either an index node or an leaf node  */
    if( this->nodes == NULL )
      return NULL;

    p_inode = & this->nodes[idx];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      /* if index node search recursively within deeper levels */
      returned_hash = hm_dissoc( & p_inode->u.inode, shift+5, hash );
      break;

    case leaf_node_type:
      /* if leaf node node remove it and move remaining nodes on its location */
      this->bitmap &= ~bitpos;
      memmove( p_inode, & this->nodes[idx+1], sizeof( hm_inode_t ) * (n-idx) );
      if( n == 1 ) {
        cul_free( this->nodes );
        this->nodes = NULL;
      }
      returned_hash = this;
      break;
    }
  }

  return returned_hash;
}


void hm_doseq( hm_t* this, int level, lambda2_t f )
{
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;
  int i;

  for( i=0; i<n; ++i )
  {
    p_inode = & this->nodes[i];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      hm_doseq( & p_inode->u.inode, level+1, f );
      break;

    case leaf_node_type:
      p_lnode = &p_inode->u.lnode;

#ifdef DEBUG
      {
        int t;
        for( t=0; t<level; ++t )
          printf("  ");
        printf("(%d) hash: %d -> %p\n", level, p_lnode->hash, p_lnode->val );
      }
#endif

      f( & p_lnode->hash, p_lnode->val );
      break;
    }
  }
}

slist_t* hm_slist( slist_t* l, hm_t* this )
{
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;
  slist_t* nl = NULL;
  int i;

  if( l == NULL )
  {
    l = slist_alloc();
    if( l == NULL )
      return NULL;
  }

  for( i=0; i<n; ++i )
  {
    p_inode = & this->nodes[i];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      nl = hm_slist( l, & p_inode->u.inode );
      break;

    case leaf_node_type:
      p_lnode = &p_inode->u.lnode;
      nl = slist_prepend( l, p_lnode );
      break;
    }

    if( nl == NULL )
      return NULL;
    l = nl;
  }

  return l;
}


void hm_free( hm_t* this, int level )
{
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  int i;

  for( i=0; i<n; ++i )
  {
    p_inode = & this->nodes[i];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      hm_free( & p_inode->u.inode, level+1 );
      break;

    case leaf_node_type:
      break;
    }
  }

  if( this->nodes )
    cul_free( this->nodes );

  if( level == 0 )
    cul_free( this );
}


void hm_free_deep( hm_t* this, int level, lambda_t free_op )
{
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;
  int i;

  if( this == NULL )
    return;

  for( i=0; i<n; ++i )
  {
    p_inode = & this->nodes[i];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      hm_free_deep( & p_inode->u.inode, level+1, free_op );
      break;

    case leaf_node_type:
      p_lnode = &p_inode->u.lnode;
      free_op( p_lnode->val );
      break;
    }
  }

  if( this->nodes )
    cul_free( this->nodes );

  if( level == 0 )
    cul_free( this );
}


hm_t* hm_assoc_with_key( hm_t* this, string_t* keystr, void* val )
{
  hm_kv_t* kv;

  if( this == NULL || keystr == NULL || val == NULL )
    return NULL;

  kv = cul_malloc( sizeof( hm_kv_t ) );
  if( kv )
  {
    kv->key = string_retain( keystr );
    if( kv->key )
    {
      kv->val = val;
      this = hm_assoc( this, 0, string_hash(keystr), kv  );
    }
    else
    {
      cul_free( kv );
    }
  }

  return this;
}

hm_t* hm_dissoc_with_key( hm_t* this, string_t* keystr )
{
  hm_leaf_node_t* node;
  hm_hash_t hash;
  hm_kv_t* kv;

  if( this == NULL || keystr == NULL )
    return NULL;

  node = hm_find( this, hash = string_hash(keystr) );
  if( node != NULL )
  {
    kv = (hm_kv_t*) ( node->val );
    if( kv )
    {
      if( kv->key )
        string_release( kv->key );
      cul_free( kv );
    }
    hm_dissoc( this, 0, hash );
  }

  return this;
}

void* hm_find_val_for_key( hm_t* this, string_t* keystr )
{
  hm_leaf_node_t* node;
  hm_kv_t* kv;
  void* v = NULL;

  if( keystr == NULL )
    return NULL;

  node = hm_find( this, string_hash(keystr) );
  if( node != NULL )
  {
    kv = (hm_kv_t*) ( node->val );
    if( kv->val )
      v = kv->val;
  }

  return v;
}

slist_t* hm_slist_with_keys( slist_t* l, hm_t* this )
{
  int n = CTPop( this->bitmap );
  hm_inode_t* p_inode;
  hm_leaf_node_t* p_lnode = NULL;
  slist_t* nl = NULL;
  int i;

  if( l == NULL )
  {
    l = slist_alloc();
    if( l == NULL )
      return NULL;
  }

  for( i=0; i<n; ++i )
  {
    p_inode = & this->nodes[i];

    switch( p_inode->typ )
    {
    case bitmap_index_node_type:
      nl = hm_slist_with_keys( l, & p_inode->u.inode );
      break;

    case leaf_node_type:
      p_lnode = &p_inode->u.lnode;
      nl = slist_prepend( l, p_lnode->val );
      break;
    }

    if( nl == NULL )
      return NULL;
    l = nl;
  }

  return l;
}


static void* free_key_in_kv( void* pkv )
{
  hm_kv_t* kv = (hm_kv_t*)pkv;

  if( kv )
  {
    if( kv->key )
      string_release( kv->key );
    cul_free( kv );
  }

  return NULL;
}

void hm_free_with_keys( hm_t* this )
{
  hm_free_deep( this, 0, free_key_in_kv );
}
