/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <limits.h>
#include <olcutils/json.h>
#include <olcutils/alloc.h>

#ifdef DEBUG
static void* print_element( void* p_e )
{
  string_t* s = p_e;

  if( s )
  {
    string_println( p_e );
    printf("----------------------------------------\n");
  }
  else
  {
    printf("________________ ERROR in %s ________________\n", __func__ );
  }
  return NULL;
}
#endif

static slist_t* json_string_split( string_t* s )
{
  char c;
  int i, splits = 0;
  int sub_start_idx = 0, sub_end_idx = 0;
  slist_t* l = slist_alloc();
  string_t* sub;
  int brace_level = 0;
  int quoted = 0;

  if( l == NULL )
    return NULL;

  for( i=0; i<s->size; ++i )
  {
    c = s->p[i];

    /*
     * jump over braced expressions and quoted strings
     */

    if( c == '{' || c == '[' )
      ++brace_level;

    if( c == '}' || c == ']' )
      --brace_level;

    if( c == '"' )
      quoted = !quoted;

    if( brace_level || quoted )
      continue;


    if( c == ',' )
    {
      /* found delimiter break, append substring to list */
      sub_end_idx = i-1;

      if( sub_end_idx >= sub_start_idx )
      {
        sub = string_retain_sub( s, sub_start_idx, sub_end_idx );
        if( sub == NULL )
          break;

        l = slist_prepend( l, string_trim( sub ) );
        if( l == NULL )
          break;

        ++splits;
      }

      sub_start_idx = i+1;
    }
  }

  /* process last argument as well */
  sub_end_idx = s->size - 1;
  if( sub_end_idx >= sub_start_idx )
  {
    sub = string_retain_sub( s, sub_start_idx, sub_end_idx );
    if( sub != NULL )
      l = slist_prepend( l, string_trim( sub ) );
  }

  return l;
}

static void* free_string_val( void* p )
{
  string_release( (string_t*)p );

  return NULL;
}

json_val_t* json_create_object( void )
{
  json_val_t* val = NULL;

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    val->u.object = hm_alloc();
    val->typ = json_object;
    if( val->u.object == NULL )
    {
      cul_free( val );
      return NULL;
    }

    return val;
  }

  return NULL;
}

json_val_t* json_create_array( void )
{
  json_val_t* val = NULL;

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    val->u.array = slist_alloc();
    val->typ = json_array;
    if( val->u.array == NULL )
    {
      cul_free( val );
      return NULL;
    }

    return val;
  }

  return NULL;
}

json_val_t* json_create_string( string_t* s )
{
  json_val_t* val = NULL;

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    val->typ = json_string;
    val->u.string = s;

    return val;
  }

  return val;
}

json_val_t* json_create_number( string_t* s )
{
  json_val_t* val = NULL;
  char numcstr[20];

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    char *ep;
    double d = strtod( string_tmp_cstring_from( s, numcstr, sizeof(numcstr) ), &ep);

    if( (ep-numcstr) == s->size )
    {
      /* all characters parsed, number seems to be ok */
      val->typ = json_number;
      val->u.number = d;
    }
    else
    {
      fprintf( stderr, "%s: could not parse number: %s\n", __func__, numcstr );
      val->typ = json_parse_error;
      val->u.error = json_invalid_number;
    }
  }

  return val;
}

json_val_t* json_create_boolean( json_boolean_t b )
{
  json_val_t* val = NULL;

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    val->typ = json_boolean;
    val->u.boolean = b;
    // printf("%d\n", (int)(val->u.boolean) );
  }

  return val;
}

json_val_t* json_create_error( json_parse_error_t e )
{
  json_val_t* val = NULL;

  val = cul_malloc( sizeof(json_val_t) );
  if( val )
  {
    val->typ = json_parse_error;
    val->u.error = e;
    // printf("%d\n", (int)(val->u.boolean) );
  }

  return val;
}

void json_free_val( json_val_t* val )
{
  if( val != NULL )
  {
    switch( val->typ )
    {
    case json_object:
      hm_free( val->u.object, 0 );
      break;

    case json_array:
      slist_free( val->u.array);
      break;

    case json_string:
      string_release( val->u.string );
      break;

    case json_number:
    case json_boolean:
    case json_parse_error:
      break;
    }

    cul_free( val );
  }
}

static json_val_t* jstr2jobject( json_val_t* o, string_t* s )
{
  slist_t* l_vk = string_split( s, ":", 2 );
  string_t *val, *key, *ckey;
  json_val_t* json_val;

  if( slist_cnt(l_vk ) == 2 )
  {
    val = slist_first( l_vk );
    key = slist_first ( slist_rest( l_vk ) );

    if( key->size>2 && key->p[0]=='"' && key->p[key->size-1]=='"' )
    {
      ckey = string_retain_sub( key, 1, key->size-2 ); /* cut out "" */
      if( ckey )
      {
        json_val = json_parse_string( val );
        if( json_val )
          hm_assoc_with_key( o->u.object, ckey, json_val  );

        string_release(ckey);
        string_release(val);
        string_release(key);
      }
    }
    else
    {
      fprintf( stderr, "%s: json key string without quotes\n", __func__ );
      json_val = json_create_error( json_unbalanced_quotes );
      if( json_val )
        hm_assoc_with_key( o->u.object, s, json_val  );

      slist_doseq( l_vk, free_string_val );
    }
  }
  else
  {
    fprintf( stderr, "%s: unbalanced json expression\n", __func__ );
    json_val = json_create_error( json_unbalanced_colon );
    if( json_val )
      hm_assoc_with_key( o->u.object, s, json_val  );

    slist_doseq( l_vk, free_string_val );
  }

  slist_free( l_vk );

  return o;
}

static json_val_t* jstr2jarray( json_val_t* a, string_t* s )
{
  json_val_t* json_val;

  json_val = json_parse_string( s );

  if( json_val )
    a->u.array = slist_prepend( a->u.array, json_val );

  return a;
}


json_val_t* json_parse_string( string_t* s )
{
  // string_println( s );
  json_val_t* val = NULL;
  char c = s->p[0], openc, closec, levelcnt = 1;
  string_t* ls;
  slist_t* elements = NULL;
  slist_t* pelements = NULL;
  int i;

  if( s->size < 1 )
    return NULL;

  if( c == '{' || c == '[' )
  {
    /* we got an compound element, object or vector */
    openc = c;
    closec = (c=='{') ? '}' : ']';

    for( i=1; i<s->size; ++i )
    {
      c = s->p[i];
      if( c == openc )
        ++levelcnt;
      if( c == closec )
        --levelcnt;

      if( levelcnt == 0 )
        break;
    }

    if( levelcnt )
    {
      fprintf( stderr, "%s: unbalanced json expression\n", __func__ );

      if( openc == '{' )
        val = json_create_error( json_unbalanced_braces );
      else
        val = json_create_error( json_unbalanced_brackets );

      return val;
    }

    ls = string_retain_sub( s, 1, i-1 );
    if( ls )
    {
      elements = json_string_split( ls );
      if( elements )
      {
        // slist_doseq( elements, print_element );
        if( s->p[0] == '{' )
        {
          /* json object */
          val = json_create_object();
          pelements = slist_prepend( elements, val );
          if( pelements )
            val = slist_reduce( pelements, (lambda2_t)jstr2jobject, NULL );
        }
        else
        {
          /* json array */
          val = json_create_array();
          pelements = slist_prepend( elements, val );
          if( pelements )
            val = slist_reduce( pelements, (lambda2_t)jstr2jarray, NULL );
        }

        slist_doseq( elements, (lambda_t)string_release );
        slist_free( pelements );
      }
      string_release( ls );
    }
  }
  else if( c == '"' )
  {
    /* create json string, cut out "" */
    if( s->p[0]=='"' && s->p[s->size-1]=='"' )
      val = json_create_string( string_retain_sub( s, 1, s->size-2 ) );
    else
      val = json_create_error( json_unbalanced_quotes );
  }
  else if( s->size == 4 && strncmp( s->p, "true", s->size ) == 0 )
  {
    /* true */
    val = json_create_boolean( json_true );
  }
  else if( s->size == 5 && strncmp( s->p, "false", s->size ) == 0 )
  {
    /* false */
    val = json_create_boolean( json_false );
  }
  else
  {
    val = json_create_number( s );
  }

  return val;
}


typedef void (*json_emerge_printfunc)( FILE*, void* );

static void json_iterate_kv( FILE*fp, void* p )
{
  hm_kv_t* kv = p;

  if( kv )
  {
    fprintf( fp, "\""); string_fprint( fp, kv->key ); fprintf( fp, "\"");
    fprintf( fp, " : ");
    json_emerge( fp, kv->val );
  }
}


static void slist_print_elements( FILE* fp, slist_t* l, json_emerge_printfunc f )
{
  while( l && l->data )
  {
    (*f)( fp, l->data );
    l = l->next;

    if( l && l->data )
      fprintf( fp, ", " );
  }
}

void json_emerge( FILE* fp, json_val_t* val )
{
  slist_t* l = NULL;

  if( val != NULL )
  {
    switch( val->typ )
    {
    case json_object:
      fprintf( fp, "{ " );
      l = hm_slist_with_keys( l, val->u.object );
      slist_print_elements( fp, l, json_iterate_kv );
      slist_free( l );
      fprintf( fp, " }" );
      break;

    case json_array:
      fprintf( fp, "[ " );
      slist_print_elements( fp, val->u.array, (json_emerge_printfunc)json_emerge );
      fprintf( fp, " ]" );
      break;

    case json_string:
      fprintf( fp, "\"" ); string_fprint( fp, val->u.string ); fprintf( fp, "\"" );
      break;

    case json_number:
      fprintf( fp, "%lg", val->u.number );
      break;

    case json_boolean:
      if( val->u.boolean ) fprintf( fp, "true"); else fprintf( fp, "false" );
      break;

    case json_parse_error:
      switch( val->u.error )
      {
      case json_unbalanced_quotes:
        fprintf( fp, "<unbalanced quotes here>" );
        break;

      case json_unbalanced_braces:
        fprintf( fp, "<unbalanced curly braces here>" );
        break;

      case json_unbalanced_brackets:
        fprintf( fp, "<unbalanced braces here>" );
        break;

      case json_unbalanced_colon:
        fprintf( fp, "<unbalanced colon here>" );
        break;

      case json_invalid_number:
        fprintf( fp, "<invalid number here>" );
        break;

      case json_unspecified_error:
        fprintf( fp, "<error while parsing expression here>" );
        break;
      }
      break;
    }
  }
}

static void json_free_kv( hm_kv_t* kv )
{
  if( kv )
  {
    if( kv->key )
      string_release( kv->key );
    if( kv->val )
      json_free( kv->val );

    cul_free( kv );
  }
}

void json_free( json_val_t* val )
{
  slist_t* l = NULL;

  if( val != NULL )
  {
    switch( val->typ )
    {
    case json_object:
      l = hm_slist_with_keys( l, val->u.object );
      slist_doseq( l, (lambda_t)json_free_kv );
      slist_free( l );
      hm_free( val->u.object, 0 );
      break;

    case json_array:
      slist_doseq( val->u.array, (lambda_t)json_free );
      slist_free( val->u.array );
      break;

    case json_string:
      string_release( val->u.string );
      break;

    case json_number:
    case json_boolean:
    case json_parse_error:
      break;
    }

    cul_free( val );
  }
}

json_val_t* json_find( json_val_t* val, string_t* key, int max_levels )
{
  slist_t *l = NULL, *i = NULL;
  json_val_t* found = NULL;

  if( val != NULL && max_levels>0 )
  {
    switch( val->typ )
    {
    case json_object:
      found = (json_val_t *)hm_find_val_for_key( val->u.object, key );

      if( !found )
      {
        i = l = hm_slist_with_keys( l, val->u.object );

        while( !found && i && i->data )
        {
          found = json_find( ((hm_kv_t*)i->data)->val, key, max_levels-1 );
          i = i->next;
        }

        slist_free( l );
      }
      break;

    case json_array:
      i = val->u.array;
      while( !found && i && i->data )
      {
        found = json_find( i->data, key, max_levels-1 );
        i = i->next;
      }
      break;

    case json_string:
      break;

    case json_number:
      break;

    case json_boolean:
      break;

    case json_parse_error:
      found = val;
      break;
    }
  }

  return found;
}

json_val_t* json_errors( json_val_t* val )
{
  string_t* search_key_empty = string_new_from( "" );
  json_val_t* found = NULL;

  if( search_key_empty )
  {
    found = json_find( val, search_key_empty, INT_MAX );
    string_release( search_key_empty );
  }

  return found;
}
