/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include <config.h>
#include <assert.h>
#include <execinfo.h>
#include <olcutils/memtrace.h>
#include <pthread.h>

#ifdef ENABLE_MEM_TRACE

static void* (*libc_malloc)(size_t) = NULL;
static void* (*libc_realloc)(void*,size_t) = NULL;
static void (*libc_free)(void*) = NULL;

static mlist_t* mlist = NULL;
static flist_t* flist = NULL;

static int trace_enabled = 0;


static pthread_mutex_t      mtrace_mutex;
static pthread_mutexattr_t  mtrace_mutex_mta;

static int memtrace_init( void )
{
  static int initialized = 0;

  if( initialized )
    return 0;

  libc_malloc = dlsym(RTLD_NEXT, "malloc" );
  libc_realloc = dlsym(RTLD_NEXT, "realloc" );
  libc_free = dlsym(RTLD_NEXT, "free" );

  if( !libc_malloc || !libc_realloc || !libc_free )
  {
    fprintf( stderr, "libc allocator functions malloc/realloc/free not found error!\n" );
    return -1;
  }

  pthread_mutexattr_init( &mtrace_mutex_mta );
  pthread_mutexattr_settype( &mtrace_mutex_mta, PTHREAD_MUTEX_RECURSIVE );
  pthread_mutex_init( &mtrace_mutex, &mtrace_mutex_mta );

  initialized = 1;

  return 0;
}


static inline backtrace_array_t get_backtrace_array( void )
{
  void* stack[MEMTRACE_MAX_BACKTRACE_ELEMENTS];
  size_t size;
  void** cb_handle = NULL;
  backtrace_array_t array;

  size = backtrace(stack, MEMTRACE_MAX_BACKTRACE_ELEMENTS);
  if( size > 0 )
  {
    assert( ( cb_handle = libc_malloc( sizeof(void *) * size ) ) );
    memcpy( cb_handle, stack, sizeof(void *) * size );
  }

  array.p = cb_handle;
  array.size = size;

  return array;
}

static inline void release_backtrace_array( backtrace_array_t* p_array )
{
  if( p_array->size > 0 )
    libc_free( p_array->p );
}


static inline void prepend_alloc_data( void* p, const size_t size )
{
  mlist_t* e = (mlist_t *)libc_malloc( sizeof(mlist_t) );
  assert( e );
  memset( e, 0, sizeof(mlist_t) );

  e->data.addr = p;
  e->data.size = size;
  e->data.bt = get_backtrace_array();

  e->next = mlist;

  mlist = e;

  // puts("---- one element added ----->\n");
}

static inline void prepend_free_data( memblck_t data )
{
  flist_t* f = (flist_t *)libc_malloc( sizeof(flist_t) );
  assert( f );
  memset( f, 0, sizeof(flist_t) );

  f->data = data;
  f->bt_free = get_backtrace_array();

  f->next = flist;

  flist = f;
}

static int remove_alloc_data( void* addr )
{
  mlist_t* l = mlist;
  mlist_t* p;

  // puts("<--- one element freed ------\n");

  if( l && l->data.addr )
  {
    if( l->data.addr == addr )
    {
#ifdef ENABLE_FREE_TRACE
      prepend_free_data( l->data );
#else
      /* remove first element in list*/
      release_backtrace_array( & l->data.bt );
#endif
      mlist = mlist->next;
      libc_free( l );
      return 0;
    }

    p = l;
    l = l->next;

    while( l && l->data.addr )
    {
      if( l->data.addr == addr )
      {
#ifdef ENABLE_FREE_TRACE
        prepend_free_data( l->data );
#else
        /* remove element in list and relink predecessor */
        release_backtrace_array( & l->data.bt );
#endif
        p->next = l->next;
        libc_free( l );
        return 0;
      }
      l = l->next;
      p = p->next;
    }
  }

  return -1; /* address not found, double free! */
}


/*!
 * enable tracing of allocator callbacks for leakage analysis
 */
void memtrace_enable( void )
{
  trace_enabled = 1;
}

/*!
 * disable tracing of allocator callbacks for leakage analysis
 */
void memtrace_disable( void )
{
  trace_enabled = 0;
}


static int mem_trace_print_mem_chunk( FILE* fp, memblck_t* p )
{
  char **strings;
  int nr_chars = 0;
  int i;

  assert( memtrace_init() == 0 );

  pthread_mutex_lock( & mtrace_mutex );

  strings = backtrace_symbols( p->bt.p, p->bt.size );
  assert( strings != NULL );

  nr_chars += fprintf( fp, "--- ADDRESS:%p, SIZE:%ld\n", p->addr, (long)p->size );
  for (i = 0; i < p->bt.size; ++i)
    fprintf( fp, "\t%s\n", strings[i] );

  libc_free( strings );

  pthread_mutex_unlock( & mtrace_mutex );

  return nr_chars;
}


/*!
 * print non-free memory chunks with allocator's backtrace
 */
int memtrace_print_log( FILE* fp )
{
  mlist_t* l = mlist;
  int nr_chars = 0;

  while( l && l->data.addr )
  {
    nr_chars += mem_trace_print_mem_chunk( fp, & l->data );
    nr_chars += fprintf( fp, "_________________________________________________________\n" );
    l = l->next;
  }

  return nr_chars;
}

/*!
 * print all freed memory chunks with free's backtrace
 */
int memtrace_print_free_log( FILE* fp )
{
  flist_t* f = flist;
  int nr_chars = 0;

  char **strings;
  int i;

  while( f && f->data.addr )
  {

    nr_chars += fprintf( fp, "Backtrace of free():\n" );
    strings = backtrace_symbols( f->bt_free.p, f->bt_free.size );
    assert( strings != NULL );
    for (i = 0; i < f->bt_free.size; ++i)
      nr_chars += fprintf( fp, "\t%s\n", strings[i] );
    libc_free( strings );

    nr_chars += fprintf( fp, "Freed memory block:\n" );
    nr_chars += mem_trace_print_mem_chunk( fp, & f->data );
    nr_chars += fprintf( fp, "_________________________________________________________\n" );
    f = f->next;
  }

  return nr_chars;
}

void* malloc( size_t size )
{
  void* addr = NULL;
  static int recursions = 0;

  assert( memtrace_init() == 0 );

  pthread_mutex_lock( & mtrace_mutex );
  ++recursions;

  // puts("__overloaded_malloc\n\t");

  if( libc_malloc )
    addr = libc_malloc( size );

  if( trace_enabled && addr && recursions==1 )
    prepend_alloc_data( addr, size );

  --recursions;
  pthread_mutex_unlock( & mtrace_mutex );

  return addr;
}

void* realloc( void* addr, size_t size )
{
  static int recursions = 0;

  assert( memtrace_init() == 0 );

  pthread_mutex_lock( & mtrace_mutex );
  ++recursions;

  // puts("__overloaded_realloc\n\t");

  if( trace_enabled )
    remove_alloc_data( addr );

  if( libc_realloc )
    addr =  libc_realloc( addr, size );

  if( trace_enabled  && addr && recursions==1 )
    prepend_alloc_data( addr, size );

  --recursions;
  pthread_mutex_unlock( & mtrace_mutex );

  return addr;
}

void free( void* addr )
{
  static int recursions = 0;

  assert( memtrace_init() == 0 );

  pthread_mutex_lock( & mtrace_mutex );
  ++recursions;

  // puts("__overloaded_free\n\t");

  if( trace_enabled && recursions==1 )
    remove_alloc_data( addr );

  if( libc_free )
    libc_free( addr );

  --recursions;
  pthread_mutex_unlock( & mtrace_mutex );
}


#else /* #ifdef ENABLE_MEM_TRACE */

/*!
 * enable tracing of allocator callbacks for leakage analysis
 */
void memtrace_enable( void )
{
  fprintf( stderr, "memtrace is not compiled, use configure option --enable-memtrace\n" );
}

/*!
 * disable tracing of allocator callbacks for leakage analysis
 */
void memtrace_disable( void )
{
  fprintf( stderr, "memtrace is not compiled, use configure option --enable-memtrace\n" );
}

/*!
 * print non-free memory chunks with allocator's backtrace
 */
int memtrace_print_log( FILE* fp )
{
  fprintf( stderr, "memtrace is not compiled, use configure option --enable-memtrace\n" );

  return -1;
}

/*!
 * print all freed memory chunks with free's backtrace
 */
int memtrace_print_free_log( FILE* fp )
{
  fprintf( stderr, "memtrace is not compiled, use configure option --enable-memtrace\n" );

  return -1;
}

#endif /* #ifdef ENABLE_MEM_TRACE */
