/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <olcutils/refcnt.h>
#include <olcutils/refstring.h>
#include <olcutils/hashfunc.h>
#include <olcutils/alloc.h>

#ifndef MIN
# define MIN(x,y) ((x)<(y)?(x):(y))
#endif


/*!
   @private internal rerpresentation of size limited string
*/
typedef struct s_lstring
{
  t_ref ref;
  int size;
  char* p;

  int chars;
  int charsize;
  // maybe
  // int encoding
  // etc. could be included but think twice before
  // doing this
} t_lstring;


/*! --- @private methods --- */

static t_lstring* lstring_new( const int size )
{
  // printf("---> lstring_new, ");
  t_lstring* s = (t_lstring*) ref_new( sizeof(t_lstring) + size );

  if( s )
  {
    s->p = (char *)s + sizeof( t_lstring );
  }

  return s;
}

static void lstring_retain( t_lstring* s )
{
  // printf("---> lstring_retain, ");
  ref_retain( (t_ref*) s );
}

static void lstring_release( t_lstring* s )
{
  // printf("---> lstring_release, ");
  ref_release( (t_ref*) s );
}



/*! --- @public methods --- */

string_t* string_new( const int size )
{
  // printf("---> string_new, ");
  string_t* s = (string_t*) ref_new( sizeof(string_t) );

  if( s )
  {
    t_lstring* l = lstring_new( size );
    if( l != NULL )
    {
      s->p_lstring = l;
      s->p = l->p;
      s->size = size;
    }
    else
    {
      cul_free( s );
      s = NULL;
    }
  }

  return s;
}

string_t* string_new_from( const char* p_cstr )
{
  const int size = strlen( p_cstr );
  string_t* s = string_new( size );

  if( s )
  {
    memcpy( s->p, p_cstr, size );
  }

  return s;
}

string_t*  string_retain( string_t* i )
{
  string_t* s = (string_t*) ref_new( sizeof(string_t) );

  if( s )
  {
    s->p_lstring = i->p_lstring;
    lstring_retain( i->p_lstring );

    s->p = i->p;
    s->size = i->size;
  }

  return s;
}

string_t* string_retain_sub( string_t* i, int begin, int end )
{
  const int size = i->size;
  string_t* s = (string_t*) ref_new( sizeof(string_t) );

  if( s && size >= begin && size >= end && end >= begin)
  {
    s->p_lstring = i->p_lstring;
    lstring_retain( i->p_lstring );

    s->p = i->p + begin;
    s->size = 1 + end - begin;
  }

  return s;
}

void string_release( string_t* s )
{
  // printf("---> string_release, ");
  lstring_release( s->p_lstring );
  ref_release( (t_ref*) s );
}

string_t* string_new_clone( string_t* i )
{
  string_t* s = string_new( i->p_lstring->size );

  if( s )
  {
    memcpy( s->p_lstring->p, i->p_lstring->p, i->p_lstring->size );
  }

  return s;
}

void string_print( const string_t* s )
{
  char format[80];
  snprintf( format, sizeof(format), "%%.%ds", s->size );
  printf( format, s->p );
}

void string_fprint( FILE* fp, const string_t* s )
{
  char format[80];
  snprintf( format, sizeof(format), "%%.%ds", s->size );
  fprintf( fp, format, s->p );
}

void string_println( const string_t* s )
{
  string_print( s );
  printf("\n");
}

const char* string_tmp_cstring_from( const string_t* s, char* c_string, const int c_string_size )
{
  if( c_string_size < s->size+1 ) {
    /* string does not fit into destination, cut it off */
    memcpy( c_string, s->p, c_string_size - 1 );
    c_string[c_string_size-1] = '\0';
  } else {
    /* string does fit into destination, copy all content */
    memcpy( c_string, s->p, s->size );
    c_string[s->size] = '\0';
  }

  return c_string;
}

#define IS_DELIM_CHAR(x) ( (x)==' ' || (x)=='\t' || (x)=='\n' || (x)=='\r' )

string_t* string_triml( string_t* s )
{
  if( s )
  {
    while( IS_DELIM_CHAR( *s->p ) && s->size > 0  )
    {
      ++(s->p);
      --(s->size);
    }
  }

  return s;
}

string_t* string_trimr( string_t* s )
{
  if( s )
  {
    while( s->size > 0  &&  IS_DELIM_CHAR( s->p[s->size-1] ) )
      --(s->size);
  }

  return s;
}

string_t* string_trim( string_t* s )
{
  return string_triml( string_trimr ( s ) );
}

uint32_t string_hash( const string_t* s )
{
  return hf_bitwisehash( s->p, s->size );
}

#define IS_DELIM_NEWLINE(x) ( (x)=='\n' || (x)=='\r' )

static int string_is_char_in_delims( const char c, const char* delims )
{
  int i;

  for( i=0; delims[i]!='\0'; ++i )
    if( c == delims[i] )
      return 1;

  return 0;
}


slist_t* string_split( string_t* s, const char* delims, int limit )
{
  char c;
  int i, splits = 0;
  int sub_start_idx = 0, sub_end_idx = 0;
  slist_t* l = slist_alloc();
  string_t* sub;

  if( l == NULL )
    return NULL;

  for( i=1; i<s->size  &&  splits < limit-1; ++i )
  {
    c = s->p[i];
    if( string_is_char_in_delims(c, delims) )
    {
      /* found delimiter break, append substring to list */
      sub_end_idx = i-1;

      if( sub_end_idx >= sub_start_idx )
      {
        sub = string_retain_sub( s, sub_start_idx, sub_end_idx );
        if( sub == NULL )
          break;

        l = slist_prepend( l, string_trim( sub ) );
        if( l == NULL )
          break;

        ++splits;
      }

      sub_start_idx = i+1;
    }
  }

  /* process last line as well */
  sub_end_idx = s->size - 1;
  if( sub_end_idx >= sub_start_idx )
  {
    sub = string_retain_sub( s, sub_start_idx, sub_end_idx );
    if( sub != NULL )
      l = slist_prepend( l, string_trim( sub ) );
  }

  return l;
}


slist_t* string_split_lines( string_t* s )
{
  return string_split( s, "\r\n", INT_MAX );
}


size_t cstr_strlcpy(char *dest, const char *src, size_t len)
{
    char *d = dest;
    char *e = dest + len; /* end of destination buffer */
    const char *s = src;

    /* Insert characters into the destination buffer
       until we reach the end of the source string
       or the end of the destination buffer, whichever
       comes first. */
    while (*s != '\0' && d < e)
        *d++ = *s++;

    /* Terminate the destination buffer, being wary of the fact
       that len might be zero. */
    if (d < e)        // If the destination buffer still has room.
        *d = 0;
    else if (len > 0) // We ran out of room, so zero out the last char
                      // (if the destination buffer has any items at all).
        d[-1] = 0;

    /* Advance to the end of the source string. */
    while (*s != '\0')
        s++;

    /* Return the number of characters
       between *src and *s,
       including *src but not including *s .
       This is the length of the source string. */
    return s - src;
}
