/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <olcutils/slist.h>
#include <olcutils/alloc.h>


slist_t* slist_alloc( void )
{
  slist_t* e = cul_malloc( sizeof(slist_t) );

  if( e )
  {
    memset( e, 0 ,sizeof(slist_t) );
  }

  return e;
}

slist_t* slist_free_first( slist_t* l )
{
  slist_t* e;

  if( l )
  {
    e = l;
    l = l->next;
    cul_free( e );
  }

  return l;
}

void slist_free( slist_t* l )
{
  while( l )
    l = slist_free_first( l );
}

void slist_free_deep( slist_t* l, lambda_t free_op )
{
  while( l )
  {
    if( l->data )
      free_op( l->data );
    l = slist_free_first( l );
  }
}

slist_t* slist_append( slist_t* l, void* data )
{
  if( l )
  {
    slist_t* e = l;

    while( e->next )
    {
      ++(e->count);
      e = e->next;
    }

    e->next = slist_alloc();
    if( e->next )
    {
      e->data = data;
      e->count = 1;
      e = e->next;
    }
    else
      l = NULL;
  }

  return l;
}

slist_t* slist_prepend( slist_t* l, void* data )
{
  slist_t* e = slist_alloc();

  if( e )
  {
    e->data = data;

    if( l )
    {
      e->count = l->count + 1;
      e->next = l;
    }
  }

  return e;
}

void* slist_first( slist_t* l )
{
  void* data = NULL;

  if( l )
  {
    data = l->data;
  }

  return data;
}

slist_t* slist_rest( slist_t* l )
{
  if( l )
  {
    l = l->next;
  }

  return l;
}

int slist_cnt( slist_t *l )
{
  int cnt = 0;

  if( l )
  {
    cnt = l->count;
  }

  return cnt;
}

int slist_empty( slist_t* l )
{
  return( l->data == NULL );
}

slist_t* slist_map( slist_t* l, lambda_t f, lambda_t ofm_free_op )
{
  slist_t* nl = slist_alloc();
  slist_t* pnl;

  if( nl )
  {
    while( l && l->data )
    {
      pnl = nl;
      nl = slist_prepend( nl, (*f)(l->data) );
      if( nl == NULL && ofm_free_op )
      {
        slist_free_deep( pnl, ofm_free_op );
        break;
      }

      l = l->next;
    }
  }

  return nl;
}

void slist_doseq( slist_t* l, lambda_t f )
{
  while( l && l->data )
  {
    (*f)(l->data);
    l = l->next;
  }
}

void* slist_reduce( slist_t* l, lambda2_t f, lambda_t ofm_free_op )
{
  void *v, *nv = NULL, *first;

  if( l )
  {
    first = v = l->data;
    l = l->next;

    while( l && l->data )
    {
      nv = f( v, l->data );

      if( ofm_free_op && v && v != first )
        ofm_free_op( v );

      if( !nv )
        break;
      v = nv;

      l = l->next;
    }
  }

  return nv;
}

static void* clone_element( void* p )
{
  return p;
}

slist_t* slist_reverse( slist_t* l )
{
  return slist_map( l, clone_element, NULL );
}
