/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <time.h>
#include <unistd.h> /* for usleep */

#include <olcutils/clist.h>


/*! size of event queue */
#define EVT_QUEUE_SIZE 50


/*! max payload size of event */
#define EVT_PAYLOAD_SIZE 20


/*!
 * number of simulated processing cycles
 * within event producer and consumer thread
 */
#define TEST_CLIST_KEEP_ALIVE_CYCLES 50


/*!
 * enum type for event identifiers
 */
typedef enum {
  EVT_1,           /*!< incomming call */
  EVT_2,           /*!< outgoing call answered */
  EVT_COUNT        /*!< < MAX number of events */
} tEvtType;


/*!
 * event data structure
 */
typedef struct  {
  t_clist           node;                    /*< For Queueing, must be first */
  tEvtType          type;                    /*< event type */
  char              data[EVT_PAYLOAD_SIZE];  /*< addititional payload data */
} tEvt;


/*!
 * state data for circular list sample application
 */
typedef struct sTestClist {
  tEvt              evtBlock[EVT_QUEUE_SIZE];
  t_clist           pool;
  t_clist           readyList;
  pthread_t         producerThread;
  pthread_t         consumerThread;
  int               producerErrors;
  int               consumerErrors;
  pthread_mutex_t   mutex;
} tTestClist;


/*!
 * generate new event with given type and payload
 */
static void put_evt( tTestClist* p, const tEvtType iEvtType, const char* data )
{
  tEvt*  pEvt;

  pthread_mutex_lock( & p->mutex );

  if ( !clist_is_empty( & p->pool ) )
  {
    // printf("get event from pool\n");
    pEvt = (tEvt *)clist_remove_head( & p->pool );
  }
  else
  {
    fprintf( stderr, "pool is empty, so get oldest event from ready list\n");
    pEvt = (tEvt *)clist_remove_head( & p->readyList );
    ++p->producerErrors;
  }

  pEvt->type = iEvtType;
  strncpy( pEvt->data, data, EVT_PAYLOAD_SIZE );

  clist_insert_tail( & p->readyList, & pEvt->node);

  pthread_mutex_unlock( & p->mutex );
}

/*!
 * fetches event from event queue.
 *
 * \return oldest event or NULL in case of queue is empty
 */
static tEvt* get_event( tTestClist* p )
{
  tEvt*  pEvt;

  pthread_mutex_lock( & p->mutex );

  if ( !clist_is_empty( & p->readyList ) )
    pEvt = (tEvt *)clist_remove_head( & p->readyList );
  else
    pEvt = NULL;

  pthread_mutex_unlock( & p->mutex );

  return pEvt;
}

/*!
 * put event back in pool
 */
static void release_event( tTestClist* p, tEvt*  pEvt )
{
  pthread_mutex_lock( & p->mutex );
  clist_insert_tail( & p->pool, & pEvt->node);
  pthread_mutex_unlock( & p->mutex );
}

/*!
 * event generator thread
 */
static void* event_producer_loop( void* pCtx )
{
  tTestClist* p = (tTestClist *) pCtx;
  int keep_alive_cnt = TEST_CLIST_KEEP_ALIVE_CYCLES;
  tEvtType nextEvt;
  char data[EVT_PAYLOAD_SIZE];

  srand(time(NULL));

  /* run for 100 cycles */
  while( keep_alive_cnt-- > 0 )
  {
    /* generate random event with carries the thread loop keep_alive_cnt
       as payload */
    nextEvt = (tEvtType)(rand() % (int)EVT_COUNT);
    snprintf( data, EVT_PAYLOAD_SIZE, "%d", keep_alive_cnt );
    printf("---> enqueue evt %d with payload: %s\n", nextEvt, data );
    put_evt( p, nextEvt, data );

    usleep( rand() % 100000 ); /* sleep for random time (0..100ms) */
  }

  return( NULL );
}


/*!
 * event consumer thread
 */
static void* event_consumer_loop( void* pCtx )
{
  tTestClist* p = (tTestClist *) pCtx;
  int keep_alive_cnt = TEST_CLIST_KEEP_ALIVE_CYCLES;
  int expected_evt_cnt = keep_alive_cnt-1;
  tEvt* pEvt;

  srand(time(NULL));

  /* run for 100 cycles */
  while( keep_alive_cnt-- > 0 )
  {
    pEvt = get_event( p );
    if( pEvt != NULL )
    {
      printf("<--- dequeued evt %d with payload: %s\n", pEvt->type, pEvt->data );
      if( atoi(pEvt->data) != expected_evt_cnt )
      {
        fprintf( stderr, "\tpayload data %d lost, got payload %s instead!\n",
                 expected_evt_cnt, pEvt->data );
        ++p->consumerErrors;
      }
      --expected_evt_cnt;

      /* do not forget to release the processed event by putting it back to the pool! */
      release_event( p, pEvt );
    }

    usleep( rand() % 100000 ); /* sleep for random time (0..100ms) */
  }

  return( NULL );
}


/*!
 * allocate memory for state struct
 */
static tTestClist* testclist_malloc (void)
{
  tTestClist* ptr = (tTestClist*) malloc( sizeof(tTestClist) );

  if (ptr)
    memset (ptr, 0, sizeof(tTestClist) );
  return ptr;
}


/*!
 * initialize state struct
 */
static int testclist_init (tTestClist* p)
{
  int retcode, i;
  void* producerThreadReturn;
  void* consumerThreadReturn;

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 )
  {
    fprintf( stderr, "Allocation of mutex failed!\n" );
    return -1;
  }

  /* initialize pool an ready ring buffers */
  clist_init( & p->pool );
  clist_init( & p->readyList );
  for (i = 0; i < EVT_QUEUE_SIZE; i++)
  {
    clist_insert_tail( & p->pool, & p->evtBlock[i].node );
  }

  retcode = pthread_create( &p->producerThread, NULL, event_producer_loop, p );
  if( retcode )
    fprintf( stderr, "Creation of generator thread failed with error %d\n", retcode );

  retcode = pthread_create( &p->consumerThread, NULL, event_consumer_loop, p );
  if( retcode )
    fprintf( stderr, "Creation of generator thread failed with error %d\n", retcode );

  /*! wait for thread completion */
  pthread_join( p->producerThread, &producerThreadReturn );
  pthread_join( p->consumerThread, &consumerThreadReturn );

  return retcode;
}

/*!
 * frees state struct
 */
static void testclist_release(tTestClist* p)
{
  if( p != NULL )
  {
    pthread_cancel( p->producerThread );
    pthread_mutex_destroy( &p->mutex );
    free( p );
  }
}

/*!
 * main test program for event queues
 */
int test_clist(void)
{
  int retcode;

  printf("\n%s()\n", __func__ );

  tTestClist* p = testclist_malloc ();
  if( p == NULL )
    return -1;

  retcode = testclist_init( p );
  if( retcode != 0 )
    return retcode;

  printf("got %d producer and conumser %d errors!\n",
        p->producerErrors, p->consumerErrors );

  retcode = p->producerErrors || p->consumerErrors;

  testclist_release(p);

  if( ! retcode )
    printf("%s()\t\t[OK]\n", __func__ );

  return retcode;
}
