/*
    cutils - Common Utilities for functional programming style under ANSI-C
    Copyright 2014 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <olcutils/hashmap.h>
#include <olcutils/slist.h>
#include <olcutils/alloc.h>

static int test_CTPop(void)
{
  unsigned long l;
  int result = 0;

  printf("testing CTPop() ...\t\t");
  fflush( stdout );

  /* break loop for(l=0;l<0x7FFFFFFF;l+=1) into two parts */
  for(l=0;l<0x7FFFFF;l+=1)
    if( CTPop_check( (int)l ) != CTPop( (int)l ) )
    {
      fprintf( stderr, "MISMATCH for %ld\n", l );
      result = -1;
      break;
    }

  if( result == 0 ) {
  for(l=0x7FFFFFFF;l>0x7FF00000;l-=1)
    if( CTPop_check( (int)l ) != CTPop( (int)l ) )
    {
      fprintf( stderr, "MISMATCH for %ld\n", l );
      result = -1;
      break;
    }
  }

  if( result )
    fprintf( stderr, "CTPop() mismatched for argument %ld error!\n", l );
  else
    printf( "[OK]\n" );

  return result;
}

static void* lambda_p_int_hash( void* key, void* val )
{
  printf("%d->%d\n", * ((int *)key), * ((int *)val) );
  return NULL;
}

static void* free_int( void* i )
{
  if( i )
    cul_free( i );
  return NULL;
}

static void* print_kiv( void* p )
{
  hm_leaf_node_t* kv = (hm_leaf_node_t*)p;
  if( kv )
    printf("%d->%d, ", kv->hash, *((int*)kv->val) );

  return kv;
}


int test_hashmap_simple(void)
{
  int result = 0;
  int a = 1;
  int b = 33;
  int c = 2;
  int d = 34;
  int e = 35;

  hm_t* my_hash = hm_alloc();;
  hm_leaf_node_t* found;
  slist_t* l = NULL;

  printf("\n%s()\n", __func__ );

  hm_assoc( my_hash, 0, 1, &a  );
  hm_dissoc( my_hash, 0, 1 );
  hm_assoc( my_hash, 0, 33, &b  );
  hm_assoc( my_hash, 0, 35, &e  );

  hm_assoc( my_hash, 0, 2, &c  );
  hm_assoc( my_hash, 0, 34, &d  );

  hm_dissoc( hm_dissoc( my_hash, 0, 35 ), 0, 34 );
  hm_assoc( my_hash, 0, 33, &e  );

  hm_doseq( my_hash, 0, lambda_p_int_hash );
  l = hm_slist( l, my_hash );
  printf("got the following list->\n");
  slist_doseq( l, print_kiv ); printf("\n");
  slist_free( l );

  found = hm_find( my_hash, 2 );
  if( found )
  {
    printf("found hash %d, val: %d\n",
           found->hash, *((int *)(found->val)) );
  }
  else
  {
    printf("hash %d not found!\n", 2 );
    result = -1;
  }

  hm_free( my_hash, 0 );

  if( ! result )
    printf("%s()\t\t[OK]\n", __func__ );

  return result;
}


int test_hashmap_dissoc(void)
{
  int result = 0;
  int a = 1;
  int b = 33;

  hm_t* my_hash = hm_alloc();;
  slist_t* l = NULL;

  printf("\n%s()\n", __func__ );

  hm_assoc( my_hash, 0, 1, &a  );
  hm_assoc( my_hash, 0, 33, &b  );

  hm_dissoc( my_hash, 0, 1 );
  hm_doseq( my_hash, 0, lambda_p_int_hash );

  l = hm_slist( l, my_hash );
  printf("got the following list->\n");
  slist_doseq( l, print_kiv ); printf("\n");
  slist_free( l );

  hm_free( my_hash, 0 );

  if( ! result )
    printf("%s()\t\t[OK]\n", __func__ );

  return result;
}


int test_hashmap_loop(void)
{
  int result = 0;
  // const int nr_elements = 65537;
  const int nr_elements = 1026;
  int i;
  int* pi;

  hm_t* my_hash = hm_alloc();
  hm_leaf_node_t* found;

  printf("\n%s()\n", __func__ );
  printf("storing %d elements now ...\n", nr_elements );

  for( i=0; i<nr_elements;++i)
  {
    pi = cul_malloc( sizeof(int) );
    if( pi != NULL )
    {
      *pi = i;
      hm_assoc( my_hash, 0, i, pi );
    }
  }

  // hm_apply( my_hash, 0, lambda_p_int_hash );

  printf("now checking content ...         \n");
  for( i=0; i<nr_elements;++i)
  {
    found = hm_find( my_hash, i );
    if( found )
    {
      if( *((int *)(found->val)) != i )
      {
        fprintf( stderr, "malformed hash value, we stored %d, but %d was found in hash!\n",
                 i, *((int *)(found->val)) );
        result = -1;
        break;
      }
    }
    else
    {
      fprintf( stderr, "hash value %d was not found error!\n", i );
      result = -1;
      break;
    }
  }

  hm_free_deep( my_hash, 0, free_int );

  if( ! result )
    printf("%s()\t\t[OK]\n", __func__ );

  return result;
}



int test_hashmap_random(void)
{
  int result = 0;
  // const int nr_elements = 1000000;
  const int nr_elements = 3000;
  int i, r, c;
  int* pr;

  slist_t* my_hashes_list = slist_alloc();
  slist_t* my_hashes_iter;
  hm_t* my_hash = hm_alloc();
  hm_leaf_node_t* found;

  printf("\n%s()\n", __func__ );
  printf("storing %d random elements now ...\n", nr_elements );

  for( i=0; i<nr_elements;++i)
  {
    r = rand();
    if( (found = hm_find( my_hash, r )) )
    {
      c = *((int*)(found->val));
      if( c == r )
      {
        /* random number already in hash, get another one */
        --i;
        // printf("\n\t\tvalue %d already hashed!\n", r);
        continue;
      }
      else
      {
        slist_t* my_hashes_list_reverse = slist_reverse( my_hashes_list );
        int* p_i;
        fprintf( stderr, "\nhash storage error occured after %d elements!\n", i );
        fprintf( stderr, "stored the following elements in hash:\n" );

        for( my_hashes_iter = my_hashes_list_reverse; !slist_empty(my_hashes_iter); my_hashes_iter = slist_rest(my_hashes_iter) )
        {
          p_i = ((int*)slist_first(my_hashes_iter));
          fprintf( stderr, "\t%d\n", *p_i );
        }

        fprintf( stderr, "\nWhen storing %d, we got %d!\n", r ,c );
        slist_free( my_hashes_list_reverse );
        slist_free_deep( my_hashes_list, free_int );

        result = -1;
        break;
      }
    }

    pr = cul_malloc( sizeof(int) );
    if( pr != NULL )
    {
      *pr = r;

      my_hashes_list = slist_prepend( my_hashes_list, pr );
      hm_assoc( my_hash, 0, r, pr );
    }

    if( !(i%100) )
    {
      printf("\t%d values hashed\r", i);
    }
  }

  /* print all stored hashes */
  // hm_apply( my_hash, 0, lambda_p_int_hash );

  printf("now checking content             ...\n");
  for( my_hashes_iter = my_hashes_list; !slist_empty(my_hashes_iter); my_hashes_iter = slist_rest(my_hashes_iter) )
  {
    r = *((int*)slist_first(my_hashes_iter));
    found = hm_find( my_hash, r );
    if( found )
    {
      if( *((int *)(found->val)) != r )
      {
        fprintf( stderr, "malformed hash value, we stored %d, but %d was found in hash!\n",
                 r, *((int *)(found->val)) );
        result = -1;
        break;
      }
    }
    else
    {
      fprintf( stderr, "hash value %d was not found error!\n", r );
      result = -1;
      break;
    }
  }

  hm_free_deep( my_hash, 0, free_int );
  slist_free( my_hashes_list );

  if( ! result )
    printf("%s()\t\t[OK]\n", __func__ );

  return result;
}

static void* lambda_p_kv( void* key, void* val )
{
  hm_kv_t* kv = (hm_kv_t *)val;

  if( kv )
  {
    printf("(");
    string_print( kv->key );
    printf(", %d), ", *((int*)kv->val) );
  }

  return NULL;
}

static void* print_kv( void* p )
{
  hm_kv_t* kv = (hm_kv_t *)p;

  if( kv )
  {
    printf("(");
    string_print( kv->key );
    printf(", %d), ", *((int*)kv->val) );
  }

  return NULL;
}

int test_hashmap_with_string_keys(void)
{
  int result = 0;

  int a = 1;
  int b = 33;
  int c = 2;
  int d = 34;
  int e = 35;

  int f = 11;

  string_t* sa = string_new_from( "1" );
  string_t* sb = string_new_from( "33" );
  string_t* sc = string_new_from( "2" );
  string_t* sd = string_new_from( "34" );
  string_t* se = string_new_from( "35" );

  hm_t* my_hash = hm_alloc();;
  int* found_pi;
  slist_t* l = NULL;

  printf("\n%s()\n", __func__ );

  hm_assoc_with_key( my_hash, sa, &a  );
  hm_assoc_with_key( my_hash, sb, &b  );
  hm_assoc_with_key( my_hash, sc, &c  );
  hm_assoc_with_key( my_hash, sd, &d  );
  hm_assoc_with_key( my_hash, se, &e  );

  hm_dissoc_with_key( my_hash, sa );
  hm_assoc_with_key( my_hash, sa, &f  );

  printf("\nstored this values in hash table->\n");
  hm_doseq( my_hash, 0, lambda_p_kv );

  l = hm_slist_with_keys( l, my_hash );
  printf("\n\ngot the following list->\n");
  slist_doseq( l, print_kv ); printf("\n");
  slist_free( l );
  printf("\n");

  found_pi = (int *)hm_find_val_for_key( my_hash, sc );
  if( found_pi && *found_pi == 2 )
  {
    printf("found val: %d\n", *found_pi );
  }
  else
  {
    printf("hash %d not found!\n", 2 );
    result = -1;
  }

  string_release( se );
  string_release( sd );
  string_release( sc );
  string_release( sb );
  string_release( sa );

  hm_free_with_keys( my_hash );

  if( ! result )
    printf("%s()\t\t[OK]\n", __func__ );

  return result;
}

int test_hashmap(void)
{
  int result = 0;

  if( ! result )
    result = test_hashmap_simple();
  if( ! result )
    result = test_hashmap_loop();
  if( ! result )
    result = test_hashmap_random();
  if( ! result )
    result = test_hashmap_dissoc();
  if( ! result )
    result = test_hashmap_with_string_keys();
  if( ! result )
    result = test_CTPop();

  return result;
}
