/*
 * vtz Valeo TrustZone driver for Qualcomm MSM9640 SoC
 *
 *   Copyright (C) 2016-2019 Valeo peiker Telematik GmbH
 *
 *   Author: Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/slab.h>		/* kzalloc() */
#include <linux/uaccess.h>	/* copy_from_user(), copy_to_user() */
#include <linux/miscdevice.h>	/* miscdev */
#include <linux/fs.h>
#include <linux/pm_runtime.h>	/* pm_runtime_get_sync() */
#include <linux/string.h>	/* memset() */
#include <linux/cdev.h>		/* cdev_init() etc. */
#include <linux/random.h>	/* prandom_bytes() */
#include <soc/qcom/scm.h>
#include <asm/cacheflush.h>	/* dmac_flush_range(), dmac_inv_range() */
#include <linux/mutex.h>
#include <linux/atomic.h>
#include <asm/barrier.h>

#ifdef CONFIG_HAS_WAKELOCK
#include <linux/wakelock.h>
#endif

#include <vtz_user.h>		/* vtz user interface */


/**********************/
/* module description */
/**********************/

#define DRV_NAME "vtz"
#define PFX DRV_NAME ": "

MODULE_AUTHOR("Simon Gleissner <simon.gleissner@valeo.com>");
MODULE_DESCRIPTION("Valeo TrustZone device driver");
MODULE_LICENSE("GPL v2");


/**************************/
/* Qualcomm trustzone API */
/**************************/

#define TZ_API_2 2
#define TZ_API_4 4

#define TZ_API TZ_API_4

/* ugly workaround for TZ 4.0 */
#if TZ_API == TZ_API_4
#define PIN_MANAGEMENT_DISABLED
#endif

//extern void pdev_set_bus(bool);		/* external clock bus setting tool in Qualcomm module pdev_test */


/**************************************************/
/* TrustZone Macros (with permission by Qualcomm) */
/**************************************************/

/* leave this as it is! */

#define TZ_SVC_CRYPTO			10	/* TZ secure monitor call service id for crypto subsystem */
#define TZ_SVC_OEM			254	/* TZ secure monitor call service id for OEM subsystem */

#if TZ_API == TZ_API_4			/* parameter format for TZ 4.0 scm_call2(), see file tz_syscall_pub.h */

#define TZ_MASK_BITS(h,l)     ((0xffffffff >> (32 - ((h - l) + 1))) << l)

#define TZ_SYSCALL_PARAM_NARGS_MASK  TZ_MASK_BITS(3,0)
#define TZ_SYSCALL_PARAM_TYPE_MASK   TZ_MASK_BITS(1,0)

#define _TZ_SYSCALL_CREATE_PARAM_ID(nargs, p1, p2, p3, p4, p5, p6, p7, p8, p9, p10, ...) \
  (((nargs)&TZ_SYSCALL_PARAM_NARGS_MASK)+ \
  (((p1)&TZ_SYSCALL_PARAM_TYPE_MASK)<<4)+ \
  (((p2)&TZ_SYSCALL_PARAM_TYPE_MASK)<<6)+ \
  (((p3)&TZ_SYSCALL_PARAM_TYPE_MASK)<<8)+ \
  (((p4)&TZ_SYSCALL_PARAM_TYPE_MASK)<<10)+ \
  (((p5)&TZ_SYSCALL_PARAM_TYPE_MASK)<<12)+ \
  (((p6)&TZ_SYSCALL_PARAM_TYPE_MASK)<<14)+ \
  (((p7)&TZ_SYSCALL_PARAM_TYPE_MASK)<<16)+ \
  (((p8)&TZ_SYSCALL_PARAM_TYPE_MASK)<<18)+ \
  (((p9)&TZ_SYSCALL_PARAM_TYPE_MASK)<<20)+ \
  (((p10)&TZ_SYSCALL_PARAM_TYPE_MASK)<<22))

#define __TZ_SYSCALL_CREATE_PARAM_ID(...) \
  _TZ_SYSCALL_CREATE_PARAM_ID(__VA_ARGS__, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0)

#define TZ_SYSCALL_CREATE_PARAM_ID_4(p1, p2, p3, p4) \
  __TZ_SYSCALL_CREATE_PARAM_ID(4, p1, p2, p3, p4)

#define TZ_SYSCALL_PARAM_TYPE_VAL              0x0
#define TZ_SYSCALL_PARAM_TYPE_BUF_RW           0x2

#define TZ_SYSCALL_CREATE_SMC_ID(o, s, f) \
  ((uint32_t)((((o & 0x3f) << 24 ) | (s & 0xff) << 8) | (f & 0xff)))

#define TZ_OWNER_SIP                        2
#define TZ_OWNER_OEM                        3
  
#define TZ_CRYPTO_SERVICE_SYM_ID                    \
  TZ_SYSCALL_CREATE_SMC_ID(TZ_OWNER_SIP, TZ_SVC_CRYPTO, 0x03)

#define TZ_CRYPTO_ASYM_SERVICE_ID                    \
  TZ_SYSCALL_CREATE_SMC_ID(TZ_OWNER_SIP, TZ_SVC_CRYPTO, 0x04)

#define TZ_OEM_TEST_ID \
  TZ_SYSCALL_CREATE_SMC_ID(TZ_OWNER_OEM, TZ_SVC_OEM, 0x01)  

#endif	/* TZ_API == TZ_API_4 */

#if TZ_API == TZ_API_4
#define TZ_SVC_SYM_ID			TZ_CRYPTO_SERVICE_SYM_ID
#define TZ_SVC_RSA_ID			TZ_CRYPTO_ASYM_SERVICE_ID
#define TZ_SVC_OEM_ID			TZ_OEM_TEST_ID
#else
#define TZ_SVC_SYM_ID			0x02		/* TZ secure monitor call command id for storage functionality */
#define TZ_SVC_RSA_ID			0x03		/* TZ secure monitor call command id for RSA functionality */
#define TZ_SVC_OEM_ID			0x01		/* TZ secure monitor call command id for OEM functionality */
#endif

#define CRYPTO_ASYM_KEY_SIZE_MAX	(512+16)	/* 4096 bits */
#define CRYPTO_ASYM_PUB_EXP_SIZE_MAX	(5)		/* 5 bytes max public exponent size supported */
#define CRYPTO_ASYM_IV_LENGTH		(16)		/* AES128 CBC IV */
#define CRYPTO_ASYM_HMAC_LENGTH		(32)		/* SHA2 will be used for HMAC  */

#define SHA256_HASH_LENGTH		(32)
#define SHA384_HASH_LENGTH		(48)
#define SHA512_HASH_LENGTH		(64)


/******************************************************************/
/* TrustZone Storage struct defines (with permission by Qualcomm) */
/******************************************************************/

typedef struct {
	void*				key_material;
	uint32_t			key_material_len;
} tz_storage_service_key_blob_t;

typedef enum {
	TZ_STORAGE_SERVICE_GENERATE_KEY	= 0x00000001,
	TZ_STORAGE_SERVICE_SEAL_DATA	= 0x00000002,
	TZ_STORAGE_SERVICE_UNSEAL_DATA	= 0x00000003
} tz_storage_service_cmd_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	tz_storage_service_key_blob_t	key_blob;
} tz_storage_service_gen_key_cmd_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	int32_t				status;
	uint32_t			key_blob_size;
} tz_storage_service_gen_key_resp_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	tz_storage_service_key_blob_t	key_blob;
	uint8_t*			plain_data;
	uint32_t			plain_dlen;
	uint8_t*			output_buffer;
	uint32_t			output_len;
} tz_storage_service_seal_data_cmd_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	int32_t				status;
	uint32_t			sealed_data_len;
} tz_storage_service_seal_data_resp_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	tz_storage_service_key_blob_t	key_blob;
	uint8_t*			sealed_data;
	uint32_t			sealed_dlen;
	uint8_t*			output_buffer;
	uint32_t			output_len;
} tz_storage_service_unseal_data_cmd_t;

typedef struct {
	tz_storage_service_cmd_t	cmd_id;
	int32_t				status;
	uint32_t			unsealed_data_len;
} tz_storage_service_unseal_data_resp_t;


/**************************************************************/
/* TrustZone RSA struct defines (with permission by Qualcomm) */
/**************************************************************/

typedef enum {
	CRYPTO_STORAGE_GENERATE_KEY_OLD		= 0x00000001,
	CRYPTO_STORAGE_EXPORT_PUBKEY		= 0x00000002,
	CRYPTO_STORAGE_SIGN_DATA_OLD		= 0x00000003,
	CRYPTO_STORAGE_VERIFY_DATA_OLD		= 0x00000004,
	CRYPTO_STORAGE_IMPORT_KEY		= 0x00000005,
	CRYPTO_STORAGE_SIGN_MHASH		= 0x00000006,
	CRYPTO_STORAGE_VERIFY_MHASH		= 0x00000007,
	CRYPTO_STORAGE_SIGN_DATA_WITH_PAD	= 0x00000008,
	CRYPTO_STORAGE_VERIFY_DATA_WITH_PAD	= 0x00000009,
	CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD	= 0x0000000A,
	CRYPTO_STORAGE_VERIFY_MHASH_WITH_PAD	= 0x0000000B,
	CRYPTO_STORAGE_LAST_CMD_ENTRY		= 0x7FFFFFFF
} crypto_asym_cmd_t;

typedef enum {
	TZ_RSA_DIGEST_PADDING_NONE	= 0x0,
	TZ_RSA_PKCS115_SHA2_256		= 0x1,
	TZ_RSA_PSS_SHA2_256		= 0x2,
	TZ_RSA_PKCS115_SHA2_384		= 0x3,
	TZ_RSA_PKCS115_SHA2_512		= 0x4
} tz_rsa_digest_pad_algo_t;

typedef struct {
	uint32_t			magic_num;
	uint32_t			version;
	tz_rsa_digest_pad_algo_t	digest_padding;
	uint8_t				modulus[CRYPTO_ASYM_KEY_SIZE_MAX];
	uint32_t			modulus_size;
	uint8_t				public_exponent[CRYPTO_ASYM_KEY_SIZE_MAX];
	uint32_t			public_exponent_size;
	uint8_t				iv[CRYPTO_ASYM_IV_LENGTH];
	uint8_t				encrypted_private_exponent[CRYPTO_ASYM_KEY_SIZE_MAX];
	uint32_t			encrypted_private_exponent_size;
	uint8_t				hmac[CRYPTO_ASYM_HMAC_LENGTH];
} crypto_rsa_key_type_t;

typedef struct {
	crypto_rsa_key_type_t		*key_material;
	uint32_t			key_material_len;
} crypto_rsa_key_blob_type_t;

typedef struct {
	uint32_t			modulus_size;
	uint64_t			public_exponent;
	tz_rsa_digest_pad_algo_t	digest_pad_type;
} crypto_asym_rsa_keygen_params_t;

typedef enum {
	CRYPTO_STORAGE_FORMAT_RAW_BYTES	= 1,
} crypto_storage_rsa_key_format_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	crypto_rsa_key_blob_type_t	key_blob;
	crypto_asym_rsa_keygen_params_t	rsa_params;
} tz_crypto_asym_gen_keypair_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	int32_t				status;
	unsigned int			key_blob_size;
} tz_crypto_asym_gen_keypair_resp_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	crypto_rsa_key_blob_type_t	key_blob;
	uint8_t*			data;
	size_t				dlen;
	uint8_t*			signeddata;
	uint32_t			signeddata_len;
} tz_crypto_asym_sign_data_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	crypto_rsa_key_blob_type_t	key_blob;
	uint8_t*			data;
	size_t				dlen;
	tz_rsa_digest_pad_algo_t	digest_pad_type;
	uint8_t*			signeddata;
	uint32_t			signeddata_len;
} tz_crypto_asym_sign_data_with_pad_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	size_t				sig_len;
	int32_t				status;
} tz_crypto_asym_sign_data_resp_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	crypto_rsa_key_blob_type_t	key_blob;
	uint8_t*			signed_data;
	size_t				signed_dlen;
	uint8_t*			signature;
	size_t				slen;
} tz_crypto_asym_verify_data_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	int32_t				status;
} tz_crypto_asym_verify_data_resp_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	crypto_rsa_key_blob_type_t	key_blob;
	crypto_storage_rsa_key_format_t	export_format;
	uint8_t*			modulus;
	uint32_t			modulus_size;
	uint8_t*			public_exponent;
	uint32_t			public_exponent_size;
} tz_crypto_storage_rsa_export_key_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	int32_t				status;
	uint32_t			modulus_size;
	uint32_t			public_exponent_size;
	uint32_t			exported_key_len;
} tz_crypto_storage_rsa_export_key_resp_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	uint8_t				modulus[CRYPTO_ASYM_KEY_SIZE_MAX];
	uint32_t			modulus_size;
	uint8_t				public_exponent[CRYPTO_ASYM_PUB_EXP_SIZE_MAX];
	uint32_t			public_exponent_size;
	uint8_t				private_exponent[CRYPTO_ASYM_KEY_SIZE_MAX];
	uint32_t			private_exponent_size;
	tz_rsa_digest_pad_algo_t	digest_pad_type;
	crypto_rsa_key_blob_type_t	key_blob;
} tz_crypto_storage_rsa_import_key_cmd_t;

typedef struct {
	crypto_asym_cmd_t		cmd_id;
	int32_t				status;
} tz_crypto_storage_rsa_import_key_resp_t;


/**************************************************************/
/* TrustZone OEM struct defines (with permission by Qualcomm) */
/**************************************************************/

typedef enum {
	TZ_OEM_PINS			= 1,
} tz_oem_cmd_t;

typedef struct {
	tz_oem_cmd_t			cmd_id;
} tz_oem_pins_cmd_t;

typedef struct {
	tz_oem_cmd_t			cmd_id;
	int32_t				status;
	uint64_t			pin_async;
	uint64_t			pin_storage;
	uint64_t			pin_reserved;
} tz_oem_pins_resp_t;


/***************************/
/* vtz internal memory api */
/***************************/

#define VTZ_MEM_ALIGN_NEON 8
#define VTZ_MEM_ALIGN_PAGE 4096
#define VTZ_MEM_BLOCK_LIMIT 2048

typedef struct {
	vtz_memory_t	complete;	/* complete allocated memory in one piece */
	vtz_memory_t	request;	/* section: request struct */
	vtz_memory_t	response;	/* section: response struct */
	vtz_memory_t	keyblob;	/* section: storage or RSA keyblob */
	vtz_memory_t	block1;		/* section: optional block (mostly input data) */
	vtz_memory_t	block2;		/* section: optional block (mostly output data) */
} vtz_mem_internal_t;


static void vtz_mem_set(vtz_memory_t *block, void* addr, size_t size)
{
	block->addr = addr;
	block->size = size;
}

static void* vtz_mem_begin(vtz_memory_t *block)
{
	return block->addr;
}

static void* vtz_mem_end(vtz_memory_t *block)
{
	uint8_t *ptr = (uint8_t*) block->addr;
	return &ptr[block->size];
}

static size_t vtz_mem_size_adjust(size_t size, size_t alignment)
{	/* for block size aligning */
	return ((size+alignment-1)&(~(alignment-1)));
}

static uint8_t* vtz_mem_page_adjust(uint8_t* addr, size_t alignment)
{	/* for block address aligning */
	return (uint8_t*)(((uintptr_t)&addr[alignment-1])&(~(alignment-1)));
}


/* blocks greater than 2048 */
/* buffer address has to be 4096 aligned */
/* buffer size has to be 4096 aligned */

static void vtz_mem_check_alignment(vtz_memory_t *block, const char* block_name)
{
	pr_info(PFX "block '%s': address=%p, size=%zu\n", block_name, block->addr, block->size);
	if(block->size > VTZ_MEM_BLOCK_LIMIT)			/* is block size more than 2k? */
	{
		if((((unsigned long)block->addr) & (VTZ_MEM_ALIGN_PAGE-1)) != 0 )	/* is block address aligned to 4k? */
		{
			pr_err(PFX "block '%s' with size %zu is unaligned\n", block_name, block->size);
		}
	}
}

static int vtz_mem_malloc(	/* returns 0 if succeeded */
	vtz_mem_internal_t	*memory_layout,
	void			**request_addr,
	size_t			request_size,
	void			**response_addr,
	size_t			response_size,
	size_t			keyblob_size,	/* can be 0 */
	size_t			block_1_size,	/* can be 0, might be > 2048 */
	size_t			block_2_size)	/* can be 0, might be > 2048 */
{
	size_t	offset_request, offset_response, offset_keyblob, offset_block_1, offset_block_2, complete_size;
	uint8_t *memory;
	bool page_align = (block_1_size > VTZ_MEM_BLOCK_LIMIT) || (block_2_size > VTZ_MEM_BLOCK_LIMIT); 

	if (!memory_layout)
		return -1;
	
	offset_request		= 0;
	offset_response		= offset_request  + vtz_mem_size_adjust(request_size,  VTZ_MEM_ALIGN_NEON);
	offset_keyblob		= offset_response + vtz_mem_size_adjust(response_size, VTZ_MEM_ALIGN_NEON);
	offset_block_1		= offset_keyblob  + vtz_mem_size_adjust(keyblob_size,  VTZ_MEM_ALIGN_NEON);
	offset_block_2		= offset_block_1  + vtz_mem_size_adjust(block_1_size,  page_align ? VTZ_MEM_ALIGN_PAGE : VTZ_MEM_ALIGN_NEON); /* ensure size alignment */
	complete_size		= offset_block_2  + vtz_mem_size_adjust(block_2_size,  page_align ? VTZ_MEM_ALIGN_PAGE : VTZ_MEM_ALIGN_NEON); /* ensure size alignment */

	if(page_align)
		complete_size += VTZ_MEM_ALIGN_PAGE;			/* size reserve for page alignment */

	memory = (uint8_t*) kzalloc(complete_size, GFP_KERNEL);		/* always aligned at 8 bytes minimums */

	if (!memory)
		return -1;

	vtz_mem_set(&memory_layout->complete,	&memory[0],			complete_size);
	vtz_mem_set(&memory_layout->request,	&memory[offset_request],	request_size);
	vtz_mem_set(&memory_layout->response,	&memory[offset_response],	response_size);
	vtz_mem_set(&memory_layout->keyblob,	&memory[offset_keyblob],	keyblob_size);

	if(page_align)
	{
		vtz_mem_set(&memory_layout->block1,	vtz_mem_page_adjust(&memory[offset_block_1], VTZ_MEM_ALIGN_PAGE),	block_1_size);
		vtz_mem_set(&memory_layout->block2,	vtz_mem_page_adjust(&memory[offset_block_2], VTZ_MEM_ALIGN_PAGE),	block_2_size);
	} else {
		vtz_mem_set(&memory_layout->block1,	&memory[offset_block_1],	block_1_size);
		vtz_mem_set(&memory_layout->block2,	&memory[offset_block_2],	block_2_size);
	}

	pr_info(PFX "block 'complete': address=%p, size=%zu\n", memory_layout->complete.addr, memory_layout->complete.size);

	vtz_mem_check_alignment(&memory_layout->request,  "request");
	vtz_mem_check_alignment(&memory_layout->response, "response");
	vtz_mem_check_alignment(&memory_layout->keyblob,  "keyblob");
	vtz_mem_check_alignment(&memory_layout->block1,   "block1");
	vtz_mem_check_alignment(&memory_layout->block2,   "block2");

	if (request_addr)
		*request_addr  = memory_layout->request.addr;
	if (response_addr)
		*response_addr = memory_layout->response.addr;

	return 0;
}

static void vtz_mem_free(vtz_mem_internal_t *memory_layout)
{
	if (memory_layout && memory_layout->complete.addr) {
		/* cleanup temporary trustzone memory to prevent leaks */
		memset(memory_layout->complete.addr, 0, memory_layout->complete.size);
		kfree(memory_layout->complete.addr);
	}
}

static int vtz_mem_copy_from_user(	/* returns 0 if succeeded */
	vtz_memory_t *kernel,
	vtz_memory_t *user)
{
	if (user->size > kernel->size)
		return -1;		/* prevent overflow */
	kernel->size = user->size;	/* set new kernelspace data size */
	return copy_from_user(kernel->addr, (void __user *)user->addr, user->size);
}

static int vtz_mem_copy_to_user(	/* returns 0 if succeeded */
	vtz_memory_t *user,
	vtz_memory_t *kernel)
{
	if (kernel->size > user->size)
		return -1;		/* prevent overflow */
	user->size = kernel->size;	/* set new userspace data size */
	return copy_to_user((void __user *)user->addr, kernel->addr, kernel->size);
}

static inline int vtz_nulltest(void * const ptr)	/* helper to prevent forbidden preprocessor warning */
{
	return (ptr==NULL);
}

#define TZ_MEM_IOCTL_INIT(KERNEL_PTR, USER_PTR)		\
	((sizeof(*KERNEL_PTR) != sizeof(*USER_PTR)) ||	\
	vtz_nulltest(KERNEL_PTR) ||			\
	vtz_nulltest(USER_PTR) ||			\
	(copy_from_user(KERNEL_PTR, USER_PTR, sizeof(*KERNEL_PTR)) != 0))

#define TZ_MEM_IOCTL_EXIT(KERNEL_PTR, USER_PTR)		\
	((sizeof(*KERNEL_PTR) != sizeof(*USER_PTR)) ||	\
	vtz_nulltest(KERNEL_PTR) ||			\
	vtz_nulltest(USER_PTR) ||			\
	(copy_to_user(USER_PTR, KERNEL_PTR, sizeof(*KERNEL_PTR)) != 0))


/********************/
/* vtz private data */
/********************/

struct vtz_device_t {
	dev_t		dev;			/* MAJOR & MINOR combined */
	struct device	*device;
	struct cdev	cdev;
	uint64_t	transaction;		/* default: 0 by kzalloc */
	uint64_t	pin;
	bool		transaction_lock;	/* default: 0 by kzalloc */
};

struct vtz_drvdata_t {
	dev_t			dev;		/* from alloc_chrdev_region() */
	struct class		*class;		/* from class_create() */

	struct vtz_device_t	dev_vdstorage;
	struct vtz_device_t	dev_vdacrypt;

	struct mutex		inside_tz_call_mutex;		/* for waiting/blocking */
	atomic_t		inside_long_tz_call_counter;	/* for returning EWOULDBLOCK */
	atomic_t		inside_short_tz_call_counter;

#ifdef CONFIG_HAS_WAKELOCK
	struct wakelock		vtz_wakelock;
#endif
};

struct vtz_drvdata_t *vtz_drvdata = NULL;	/* private driver data */


/***************************/
/* scm call into trustzone */
/***************************/

#if TZ_API == TZ_API_2

static int vtz_scm_call(uint32_t scm_service_id, uint32_t scm_cmd_id, vtz_mem_internal_t *tz_memory)
{
	uint32_t scm_cmd_buf[4];
	int rc;

	scm_cmd_buf[0] = virt_to_phys(tz_memory->request.addr);
	scm_cmd_buf[1] = tz_memory->request.size;
	scm_cmd_buf[2] = virt_to_phys(tz_memory->response.addr);
	scm_cmd_buf[3] = tz_memory->response.size;

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock(&vtz_drvdata->vtz_wakelock);
#endif

	dmac_flush_range(vtz_mem_begin(&tz_memory->complete), vtz_mem_end(&tz_memory->complete));	/* flush write cache */
	/* pdev_set_bus(1); */
	rc = scm_call(scm_service_id, scm_cmd_id, &scm_cmd_buf, sizeof(scm_cmd_buf), NULL, 0);		/* call trustzone */
	/* pdev_set_bus(0); */
	dmac_inv_range(vtz_mem_begin(&tz_memory->complete), vtz_mem_end(&tz_memory->complete));		/* invalidate read cache */

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_timeout(&vtz_drvdata->vtz_wakelock, HZ / 4);	/* 250ms delay */
/*	wake_unlock(&vtz_drvdata->vtz_wakelock); */
#endif

	return rc;
}

#elif TZ_API == TZ_API_4

static int vtz_scm_call(uint32_t scm_service_id, uint32_t scm_cmd_id, vtz_mem_internal_t *tz_memory)
{
	struct scm_desc desc = {0};
	int rc;

	desc.arginfo = TZ_SYSCALL_CREATE_PARAM_ID_4(
		TZ_SYSCALL_PARAM_TYPE_BUF_RW,	/* args[0] */
		TZ_SYSCALL_PARAM_TYPE_VAL,	/* args[1] */
		TZ_SYSCALL_PARAM_TYPE_BUF_RW,	/* args[2] */
		TZ_SYSCALL_PARAM_TYPE_VAL);	/* args[3] */
	desc.args[0] = virt_to_phys(tz_memory->request.addr);
	desc.args[1] = tz_memory->request.size;
	desc.args[2] = virt_to_phys(tz_memory->response.addr);
	desc.args[3] = tz_memory->response.size;

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock(&vtz_drvdata->vtz_wakelock);
#endif

	dmac_flush_range(vtz_mem_begin(&tz_memory->complete), vtz_mem_end(&tz_memory->complete));	/* flush write cache */
	//pdev_set_bus(1);										/* enable crypto clocks */
	rc = scm_call2(scm_cmd_id, &desc);								/* call trustzone */
	//pdev_set_bus(0);										/* disable crypto clocks */
	dmac_inv_range(vtz_mem_begin(&tz_memory->complete), vtz_mem_end(&tz_memory->complete));		/* invalidate read cache */

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_timeout(&vtz_drvdata->vtz_wakelock, HZ / 4);	/* 250ms delay */
#endif

	return rc;
}

#else
#error "undefined Qualcomm TrustZone API version"
#endif


/*********************/
/* error exit helper */
/*********************/

static int vtz_throw_exception(int rc, vtz_mem_internal_t *tz_memory_cleanup, const char* const funcname, const int line)
{
	vtz_mem_free(tz_memory_cleanup);				/* NULL check is done inside in vtz_mem_free */
	pr_err(PFX "exception %d in %s() at line %d\n", rc, funcname, line);
	return rc;
}


/************************************/
/* transaction random number helper */
/************************************/

static void vtz_next_transaction(uint64_t* transaction_ptr)
{									/* for now, use the kernel PRNG */
	prandom_bytes(transaction_ptr, sizeof(uint64_t));		/* pseudo random numbers, do not consume entrophy */
}


/*********************************************/
/* ioctl functions for registering a device  */
/*********************************************/

static int vtz_register(						/* returns 0 if succeeded */
	struct vtz_device_t		*private_device_data,
	vtz_ioctl_register_t __user	*ioctl_data_user)
{
	vtz_ioctl_register_t		ioctl_data_kernel;		/* stores kernel copy of ioctl user argument */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

	if(private_device_data->transaction_lock == 0)			/* default when run first: 0 */
	{
		private_device_data->transaction_lock = (ioctl_data_kernel.transaction_challenge != 0);
	}
	else								/* device had already been opened and is locked */
	{								/* test transaction code */
#ifndef PIN_MANAGEMENT_DISABLED
		if(private_device_data->transaction != ioctl_data_kernel.transaction_challenge)
			return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* device is locked and transaction is wrong */
#endif
	}

	vtz_next_transaction(&private_device_data->transaction);	/* get transaction challenge */
	private_device_data->transaction |= 1;				/* set lowest bit for challenge multiplication*/
	ioctl_data_kernel.transaction_challenge = private_device_data->transaction;
	private_device_data->transaction *= private_device_data->pin;	/* calculate challenge */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);

	return 0;
}


/**************************************/
/* ioctl functions for secure storage */
/**************************************/

static int vtz_storage_generate(					/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_generate_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_generate_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_gen_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_gen_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			VTZ_STORAGE_KEYBLOB_SIZE,			/* maximum keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	request->cmd_id = TZ_STORAGE_SERVICE_GENERATE_KEY;		/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_GENERATE_KEY)	/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.keyblob.size = response->key_blob_size;		/* get response data, adjust keyblob size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_storage_seal(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_seal_unseal_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_seal_unseal_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_seal_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_seal_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.unsealed.size,		/* uncrypted data buffer */
			ioctl_data_kernel.sealed.size))			/* crypted data buffer */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.unsealed))	/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	request->cmd_id = TZ_STORAGE_SERVICE_SEAL_DATA;			/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->plain_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);
	request->plain_dlen = tz_memory.block1.size;
	request->output_buffer = (uint8_t*)virt_to_phys(tz_memory.block2.addr);
	request->output_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_SEAL_DATA)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sealed_data_len;		/* get response data, adjust sealed data size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.sealed, &tz_memory.block2))	/* copy kernel sealed data into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_storage_unseal(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_seal_unseal_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_seal_unseal_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_unseal_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_unseal_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.sealed.size,			/* crypted data buffer */
			ioctl_data_kernel.unsealed.size))		/* uncrypted data buffer */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.sealed))	/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	request->cmd_id = TZ_STORAGE_SERVICE_UNSEAL_DATA;		/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->sealed_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);
	request->sealed_dlen = tz_memory.block1.size;
	request->output_buffer = (uint8_t*)virt_to_phys(tz_memory.block2.addr);
	request->output_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_UNSEAL_DATA)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->unsealed_data_len;		/* get response data, adjust unsealed data size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.unsealed, &tz_memory.block2))	/* copy kernel unsealed data into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}


/**********************************/
/* ioctl functions for RSA crypto */
/**********************************/

static int vtz_rsa_generate(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_generate_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_generate_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_gen_keypair_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_gen_keypair_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			sizeof(crypto_rsa_key_type_t),			/* keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	request->cmd_id = CRYPTO_STORAGE_GENERATE_KEY_OLD;		/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->rsa_params.modulus_size = ioctl_data_kernel.modulus_bitlen;
	request->rsa_params.public_exponent = ioctl_data_kernel.public_exponent;
	request->rsa_params.digest_pad_type = ioctl_data_kernel.digest_pad_type;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_GENERATE_KEY_OLD)	/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.keyblob.size = response->key_blob_size;		/* get response data, adjust keyblob size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT ,&tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}


static int vtz_rsa_sign(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_sign_data_cmd_t		*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_sign_data_resp_t		*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.data.size,			/* data storage size */
			ioctl_data_kernel.signature.size))		/* signature storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_SIGN_DATA_OLD;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->dlen = tz_memory.block1.size;
	request->signeddata = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->signeddata_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_SIGN_DATA_OLD)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sig_len;			/* get response data, adjust signature size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.signature, &tz_memory.block2))	/* copy kernel signature into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_verify(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_verify_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_verify_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.data.size,			/* data storage size */
			ioctl_data_kernel.signature.size))		/* signature storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block2, &ioctl_data_kernel.signature))	/* copy userspace signature into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_VERIFY_DATA_OLD;		/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->signed_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->signed_dlen = tz_memory.block1.size;
	request->signature = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->slen = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_VERIFY_DATA_OLD)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_export(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_export_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_export_t			ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_storage_rsa_export_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_storage_rsa_export_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */
	uint8_t* public_exponent_ptr;
	uint32_t i;

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.modulus.size,			/* modulus storage size */
			sizeof(ioctl_data_kernel.public_exponent)))	/* public exponent storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_EXPORT_PUBKEY;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->export_format = CRYPTO_STORAGE_FORMAT_RAW_BYTES;
	request->modulus = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->modulus_size = tz_memory.block1.size;
	request->public_exponent = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->public_exponent_size = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_EXPORT_PUBKEY)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.block1.size = response->modulus_size;
	if (vtz_mem_copy_to_user(&ioctl_data_kernel.modulus, &tz_memory.block1))	/* copy kernel modulus into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	if(response->public_exponent_size > sizeof(ioctl_data_kernel.public_exponent))
		return vtz_throw_exception(-EOVERFLOW, &tz_memory, __func__, __LINE__);

	ioctl_data_kernel.public_exponent = 0;
	public_exponent_ptr = (uint8_t*)tz_memory.block2.addr;
	for(i=response->public_exponent_size; i>0; i--)
		ioctl_data_kernel.public_exponent = (ioctl_data_kernel.public_exponent << 8) | (uint64_t)(*public_exponent_ptr++);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_sign_mhash(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_sign_data_with_pad_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_sign_data_resp_t		*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.data.size,			/* hash size */
			ioctl_data_kernel.signature.size))		/* signature storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace hash into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD;		/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->dlen = tz_memory.block1.size;
	request->signeddata = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->signeddata_len = tz_memory.block2.size;

	switch(tz_memory.block1.size)					/* identify padding type by hash buffer size */
	{
		case SHA256_HASH_LENGTH:
			request->digest_pad_type = TZ_RSA_PKCS115_SHA2_256;
			break;
		case SHA384_HASH_LENGTH:
			request->digest_pad_type = TZ_RSA_PKCS115_SHA2_384;
			break;
		case SHA512_HASH_LENGTH:
			request->digest_pad_type = TZ_RSA_PKCS115_SHA2_512;
			break;
		default:
			return vtz_throw_exception(-EMSGSIZE, &tz_memory, __func__, __LINE__);
	}

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD)	/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sig_len;			/* get response data, adjust signature size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.signature, &tz_memory.block2))	/* copy kernel signature into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_import(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_import_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_import_t			ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_storage_rsa_import_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_storage_rsa_import_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */
	uint32_t i;

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, __func__, __LINE__);	/* error with user memory */

#ifndef PIN_MANAGEMENT_DISABLED
	if(ioctl_data_kernel.transaction != private_device_data->transaction)
		return vtz_throw_exception(-EPERM, NULL, __func__, __LINE__);	/* transaction is wrong */
#endif

	if((ioctl_data_kernel.public_exponent > 0xFFFFFFFFFFLL) ||	/* check for input buffer overflows & wrong arguments */
			(ioctl_data_kernel.public_exponent == 0) ||
			(ioctl_data_kernel.modulus.size > CRYPTO_ASYM_KEY_SIZE_MAX) ||
			(ioctl_data_kernel.private_exponent.size > CRYPTO_ASYM_KEY_SIZE_MAX))
		return vtz_throw_exception(-EOVERFLOW, NULL, __func__, __LINE__);	/* error with input buffers */

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	request->cmd_id = CRYPTO_STORAGE_IMPORT_KEY;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;

	/* copy modulus and private exponent from user mode to tz struct */
	if(copy_from_user(&request->modulus[0], ioctl_data_kernel.modulus.addr, ioctl_data_kernel.modulus.size))
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	request->modulus_size = ioctl_data_kernel.modulus.size;
	if(copy_from_user(&request->private_exponent[0], ioctl_data_kernel.private_exponent.addr, ioctl_data_kernel.private_exponent.size))
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);
	request->private_exponent_size = ioctl_data_kernel.private_exponent.size;

	/* copy public exponent to tz struct (force big endian) */
	for(i = CRYPTO_ASYM_PUB_EXP_SIZE_MAX; i > 0; i--)		/* 5..1, search highest byte != 0 */
		if((ioctl_data_kernel.public_exponent & (0xFFLL << (8*(i-1)))) != 0) /* 0xFF00000000, 0xFF000000, 0xFF0000, 0xFF00, 0xFF */
			break;
	request->public_exponent_size = i;
	for(; i > 0; i--)						/* x..1, copy only used bytes */
		request->public_exponent[request->public_exponent_size - i] = (uint8_t)(ioctl_data_kernel.public_exponent >> (8*(i-1)));

	request->digest_pad_type = ioctl_data_kernel.digest_pad_type;	/* set padding type */

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_IMPORT_KEY)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT ,&tz_memory, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}


/*********************************************/
/* EWOULDBLOCK & mutex locking inside ioctrl */
/*********************************************/

/* atomic counter helper */

static void vtz_short_tz_duration_count_inc(void)
{
	atomic_inc(&vtz_drvdata->inside_short_tz_call_counter);
	smp_mb__after_atomic();
}

static void vtz_short_tz_duration_count_dec(void)
{
	atomic_dec(&vtz_drvdata->inside_short_tz_call_counter);
	smp_mb__after_atomic();
}

static int vtz_short_tz_duration_running(void)
{
	return (atomic_read(&vtz_drvdata->inside_short_tz_call_counter) != 0);
}

static void vtz_long_tz_duration_count_inc(void)
{
	atomic_inc(&vtz_drvdata->inside_long_tz_call_counter);
	smp_mb__after_atomic();
}

static void vtz_long_tz_duration_count_dec(void)
{
	atomic_dec(&vtz_drvdata->inside_long_tz_call_counter);
	smp_mb__after_atomic();
}

static int vtz_long_tz_duration_running(void)
{
	return (atomic_read(&vtz_drvdata->inside_long_tz_call_counter) != 0);
}

/* locking api */

static int vtz_enter_critical_short_tz_duration(void)
{
	vtz_short_tz_duration_count_inc();		/* announce short tz duration call */
							/* now long tz duration calls will not block us */
	if(vtz_long_tz_duration_running()) {		/* query long tz duration call */
		vtz_short_tz_duration_count_dec();	/* cleanup */
		return -EWOULDBLOCK;			/* long tz duration call active */
	}

	mutex_lock(&vtz_drvdata->inside_tz_call_mutex);	/* short call already announced, we can wait */
	return 0;					/* success */
}

static void vtz_leave_critical_short_tz_duration(void)
{
	vtz_short_tz_duration_count_dec();
	mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);
}

static void vtz_enter_critical_long_tz_duration(void)
{
	vtz_long_tz_duration_count_inc();				/* announce long tz duration call, now short tz duration calls will not be blocked */
	mutex_lock(&vtz_drvdata->inside_tz_call_mutex);			/* wait for exclusive tz access */

	while(vtz_short_tz_duration_running()) {			/* are short tz duration calls active? we might need several tries to aquire exclusive access */
		mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);	/* we must not block for a long time now, so try again later, */
		schedule();						/* let other processes take mutex first */
		mutex_lock(&vtz_drvdata->inside_tz_call_mutex);		/* wait again for exclusive tz access */
	}
}

static void vtz_leave_critical_long_tz_duration(void)
{
	vtz_long_tz_duration_count_dec();
	mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);
}


/**********************/
/* file i/o vdstorage */
/**********************/

static int vtz_storage_file_open(struct inode *inode, struct file *file)
{
	struct vtz_drvdata_t *private_driver_data = vtz_drvdata;

	if (!private_driver_data)
		return -ENODATA;

	if (inode->i_rdev != private_driver_data->dev_vdstorage.dev)
		return -ENXIO;

	if (file->private_data)		/* device already open? */
		return -EBADF;

	file->private_data = (void*) &private_driver_data->dev_vdstorage;

	return nonseekable_open(inode, file);
}

static int vtz_storage_file_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}

static long vtz_storage_file_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct vtz_device_t *private_device_data = (struct vtz_device_t*) file->private_data;
	long rc;

	if(!private_device_data)
		return -ENODATA;

	rc = vtz_enter_critical_short_tz_duration();
	if(rc)
		return rc;

	switch (cmd) {
	case VTZ_IOCTL_STORAGE_REGISTER:
		rc = vtz_register(private_device_data, (vtz_ioctl_register_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_GENERATE:
		rc = vtz_storage_generate(private_device_data, (vtz_ioctl_storage_generate_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_SEAL:
		rc = vtz_storage_seal(private_device_data, (vtz_ioctl_storage_seal_unseal_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_UNSEAL:
		rc = vtz_storage_unseal(private_device_data, (vtz_ioctl_storage_seal_unseal_t __user *)arg);
		break;
	default:
		rc = -ENOTTY;
		break;
	}

	vtz_leave_critical_short_tz_duration();
	return rc;
}

static const struct file_operations vtz_fops_storage = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl	= vtz_storage_file_ioctl,
	.open		= vtz_storage_file_open,
	.release	= vtz_storage_file_release,
	.llseek		= no_llseek,
};


/*********************/
/* file i/o vdacrypt */
/*********************/

static int vtz_acrypt_file_open(struct inode *inode, struct file *file)
{
	struct vtz_drvdata_t *private_driver_data = vtz_drvdata;

	if (!private_driver_data)
		return -ENODATA;

	if (inode->i_rdev != private_driver_data->dev_vdacrypt.dev)
		return -ENXIO;

	if (file->private_data)		/* device already open? */
		return -EBADF;

	file->private_data = (void*) &private_driver_data->dev_vdacrypt;

	return nonseekable_open(inode, file);
}

static int vtz_acrypt_file_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}

static long vtz_acrypt_file_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct vtz_device_t *private_device_data = (struct vtz_device_t*) file->private_data;
	long rc;

	if(!private_device_data)
		return -ENODATA;

	if(cmd == VTZ_IOCTL_RSA_GENERATE) {
		vtz_enter_critical_long_tz_duration();
	} else {
		rc = vtz_enter_critical_short_tz_duration();
		if(rc)
			return rc;
	}

	switch (cmd) {
	case VTZ_IOCTL_RSA_REGISTER:
		rc = vtz_register(private_device_data, (vtz_ioctl_register_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_GENERATE:
		rc = vtz_rsa_generate(private_device_data, (vtz_ioctl_rsa_generate_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_SIGN:
		rc = vtz_rsa_sign(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_VERIFY:
		rc = vtz_rsa_verify(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_EXPORT:
		rc = vtz_rsa_export(private_device_data, (vtz_ioctl_rsa_export_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_SIGN_MHASH:
		rc = vtz_rsa_sign_mhash(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_IMPORT:
		rc = vtz_rsa_import(private_device_data, (vtz_ioctl_rsa_import_t __user *)arg);
		break;
	default:
		rc = -ENOTTY;
		break;
	}

	if(cmd == VTZ_IOCTL_RSA_GENERATE)
		vtz_leave_critical_long_tz_duration();
	else
		vtz_leave_critical_short_tz_duration();
	return rc;
}

static const struct file_operations vtz_fops_acrypt = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl	= vtz_acrypt_file_ioctl,
	.open		= vtz_acrypt_file_open,
	.release	= vtz_acrypt_file_release,
	.llseek		= no_llseek,
};


/*****************************/
/* device construct/destroy  */
/*****************************/

static int __init vtz_device_construct(
	struct vtz_device_t		*vtz_device,
	struct class			*class,
	const char			*devicename,
	const struct file_operations	*fops,
	dev_t				dev)
{
	struct device *device;
	int rc;

	BUG_ON(vtz_device == NULL || class == NULL || devicename == NULL || fops == NULL );

	cdev_init(&vtz_device->cdev, fops);
	vtz_device->cdev.owner = THIS_MODULE;
	rc = cdev_add(&vtz_device->cdev, dev, 1);
	if (rc) {
		pr_err(PFX "vtz_device_construct(): cdev_add() failed for /dev/%s\n", devicename);
		return rc;
	}

	vtz_device->dev = dev;	/* set if cdev_add() succeeded, 0 instead (by kzalloc) */

	device = device_create(class, NULL, dev, NULL, devicename );
	if (IS_ERR(device)) {
		rc = PTR_ERR(device);
		pr_err(PFX "vtz_device_construct(): device_create() failed for /dev/%s\n", devicename);
		cdev_del(&vtz_device->cdev);
		return rc;
	}

	vtz_device->device = device;	/* set if device_create() succeeded, 0 instead (by kzalloc) */
	return 0;
}

static int vtz_device_destroy(struct vtz_device_t *vtz_device, struct class *class)
{
	BUG_ON(vtz_device == NULL);

	if (vtz_device->dev) {			/* 0 if not initialized (default by kzalloc) */
		if (class && vtz_device->device) {	/* NULL if not initialized (default by kzalloc) */
			device_destroy(class, vtz_device->dev);
		}
		cdev_del(&vtz_device->cdev);
	}
	return 0;
}

/******************/
/* pin management */
/******************/

#ifndef PIN_MANAGEMENT_DISABLED

static int vtz_pin_management(uint64_t *pin_vdstorage, uint64_t *pin_vdacrypt)
{
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_oem_pins_cmd_t			*request;		/* trustzone request (inside tz_memory) */
	tz_oem_pins_resp_t			*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			0, 0, 0))					/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, __func__, __LINE__);	/* No buffer space available */

	request->cmd_id = TZ_OEM_PINS;					/* set request data */

	rc = vtz_scm_call(TZ_SVC_OEM, TZ_SVC_OEM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_OEM_PINS)				/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, __func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, __func__, __LINE__);	/* positive return value! */

	*pin_vdstorage  = response->pin_storage;			/* no pins in open source code :-( */
	*pin_vdacrypt   = response->pin_async;

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

#endif

/******************/
/* driver control */
/******************/

static void vtz_driver_exit(void)
{
	if (vtz_drvdata) {
#ifdef CONFIG_HAS_WAKELOCK
		wake_lock_destroy(&vtz_drvdata->vtz_wakelock);
#endif
		vtz_device_destroy(&vtz_drvdata->dev_vdstorage,  vtz_drvdata->class);
		vtz_device_destroy(&vtz_drvdata->dev_vdacrypt,   vtz_drvdata->class);

		if (vtz_drvdata->class)
			class_destroy(vtz_drvdata->class);

		if (vtz_drvdata->dev)
			unregister_chrdev_region(vtz_drvdata->dev,3);

		mutex_destroy(&vtz_drvdata->inside_tz_call_mutex);

		kfree(vtz_drvdata);
		vtz_drvdata = NULL;
	}
}

static int __init vtz_driver_init(void)
{
	int rc = 0;

	if (vtz_drvdata) {
		pr_err(PFX "vtz_driver_init(): device already initialized\n");
		return -EBUSY;
	}

	/* allocate private device memory, set to zero (kzalloc) !! */
	vtz_drvdata = kzalloc(sizeof(struct vtz_drvdata_t), GFP_KERNEL);
	if (!vtz_drvdata) {
		pr_err(PFX "vtz_driver_init(): kzalloc() failed\n");
		return -ENOMEM;
	}

	mutex_init(&vtz_drvdata->inside_tz_call_mutex);
	atomic_set(&vtz_drvdata->inside_short_tz_call_counter, 0);
	atomic_set(&vtz_drvdata->inside_long_tz_call_counter, 0);
	smp_mb__after_atomic();

	/* allocate device id range for three devices */
	rc = alloc_chrdev_region(&vtz_drvdata->dev, 0, 3, DRV_NAME);
	if (rc) {
		pr_err(PFX "vtz_driver_init(): alloc_chrdev_region() failed\n");
		vtz_drvdata->dev = 0;
		vtz_driver_exit();
		return rc;
	}

	/* create device class */
	vtz_drvdata->class = class_create(THIS_MODULE, DRV_NAME);
	if (IS_ERR(vtz_drvdata->class)) {
		pr_err(PFX "vtz_driver_init(): class_create() failed\n");
		rc = PTR_ERR(vtz_drvdata->class);
		vtz_drvdata->class = NULL;
		vtz_driver_exit();
		return rc;
	}

	/* create device /dev/vdstorage */
	rc = vtz_device_construct(
		&vtz_drvdata->dev_vdstorage,
		vtz_drvdata->class,
		"vtzstorage",
		&vtz_fops_storage,
		MKDEV(MAJOR(vtz_drvdata->dev), 0));
	if (rc) {
		vtz_driver_exit();
		return rc;
	}

	/* create device /dev/vdacrypt */
	rc = vtz_device_construct(
		&vtz_drvdata->dev_vdacrypt,
		vtz_drvdata->class,
		"vtzacrypt",
		&vtz_fops_acrypt,
		MKDEV(MAJOR(vtz_drvdata->dev), 1));
	if (rc) {
		vtz_driver_exit();
		return rc;
	}

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_init(&vtz_drvdata->vtz_wakelock, WAKE_LOCK_IDLE, "vtz");
#endif

#ifndef PIN_MANAGEMENT_DISABLED
	/* get challenge pins out of trustzone (sorry) */
	rc = vtz_pin_management(
		&vtz_drvdata->dev_vdstorage.pin,
		&vtz_drvdata->dev_vdacrypt.pin);
	if (rc) {
		vtz_driver_exit();
		return rc;
	}
#endif

	pr_info(PFX "initialized\n");

	return 0;
}


/******************/
/* module control */
/******************/

static int __init vtz_module_init(void)
{
	return vtz_driver_init();
}

static void __exit vtz_module_exit(void)
{
	vtz_driver_exit();
}

module_init(vtz_module_init);
module_exit(vtz_module_exit);
