/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>		/*__init and __exit*/
#include <linux/cdev.h>		/* cdev_init() etc. */
#include <linux/device.h> 
#include <linux/types.h>		 /*atomic_t*/
#include <linux/slab.h>		/* kzalloc() */
#include <linux/miscdevice.h>	/* miscdev struct device*/
#include <linux/fs.h>		 /*file ops*/
#include <linux/mutex.h>		 /*struct mutex */

#include <linux/atomic.h> /* atomic operations*/
#include <asm/barrier.h> /*smp_mb*/ /*included here and in vtz2_helper.h*/

#ifdef CONFIG_HAS_WAKELOCK
#include <linux/wakelock.h> /*included here and in vtz2_scm.h*/
#endif

#include <vtz2_main.h>
#include <vtz_user.h>		/* vtz user interface */
#include <vtz2_helper.h>

#ifdef CONFIG_VTZ2_STORAGE
	#include <vtz2_storage.h>		 /*vtz2 storage interfaces */
#endif

#include <vtz2_scm.h>

#ifdef CONFIG_VTZ2_RSA
	#include <vtz2_rsa.h>
#endif

#ifdef CONFIG_VTZ2_LOG
	#include <vtz2_log.h>
#endif

/**********************/
/* module description */
/**********************/

MODULE_AUTHOR("Mohamed Ahmed-Hassan <mohamed.ahmed-hassan@valeo.com>");
MODULE_DESCRIPTION("Valeo TrustZone2 device driver");
MODULE_LICENSE("GPL v2");

/********************/
/* vtz private data */
/********************/

struct vtz_drvdata_t *vtz_drvdata = NULL;	/* private driver data */

/*****************************/
/* device construct/destroy  */
/*****************************/

static int __init vtz_device_construct(
	struct vtz_device_t		*vtz_device,
	struct class			*class,
	const char			*devicename,
	const struct file_operations	*fops,
	dev_t				dev)
{
	struct device *device;
	int rc;

	BUG_ON(vtz_device == NULL || class == NULL || devicename == NULL || fops == NULL );

	cdev_init(&vtz_device->cdev, fops);
	vtz_device->cdev.owner = THIS_MODULE;
	rc = cdev_add(&vtz_device->cdev, dev, 1);
	if (rc) {
		pr_err(PFX "vtz_device_construct(): cdev_add() failed for /dev/%s\n", devicename);
		return rc;
	}

	vtz_device->dev = dev;	/* set if cdev_add() succeeded, 0 instead (by kzalloc) */

	device = device_create(class, NULL, dev, NULL, devicename );
	if (IS_ERR(device)) {
		rc = PTR_ERR(device);
		pr_err(PFX "vtz_device_construct(): device_create() failed for /dev/%s\n", devicename);
		cdev_del(&vtz_device->cdev);
		return rc;
	}

	vtz_device->device = device;	/* set if device_create() succeeded, 0 instead (by kzalloc) */
	return 0;
}

static int vtz_device_destroy(struct vtz_device_t *vtz_device, struct class *class)
{
	BUG_ON(vtz_device == NULL);

	if (vtz_device->dev) {			/* 0 if not initialized (default by kzalloc) */
		if (class && vtz_device->device) {	/* NULL if not initialized (default by kzalloc) */
			device_destroy(class, vtz_device->dev);
		}
		cdev_del(&vtz_device->cdev);
	}
	return 0;
}

 /******************/
/* driver control */
/******************/

static void vtz_driver_exit(void)
{
	if (vtz_drvdata) {
	#ifdef CONFIG_HAS_WAKELOCK
			wake_lock_destroy(&vtz_drvdata->vtz_wakelock);
	#endif
	#ifdef CONFIG_VTZ2_STORAGE
			vtz_device_destroy(&vtz_drvdata->dev_vdstorage,  vtz_drvdata->class);
	#endif
	#ifdef CONFIG_VTZ2_RSA
		vtz_device_destroy(&vtz_drvdata->dev_vdacrypt,   vtz_drvdata->class);
	#endif
	
	#ifdef CONFIG_VTZ2_LOG
			vtz_device_destroy(&vtz_drvdata->dev_vdlog,   vtz_drvdata->class);
	#endif
		if (vtz_drvdata->class)
			class_destroy(vtz_drvdata->class);

		if (vtz_drvdata->dev)
			unregister_chrdev_region(vtz_drvdata->dev,DEVICES_COUNT);

		mutex_destroy(&vtz_drvdata->inside_tz_call_mutex);
		mutex_destroy(&vtz_drvdata->log_buffer_mutex);

		kfree(vtz_drvdata);
		vtz_drvdata = NULL;
	}
}

static int __init vtz_driver_init(void)
{
	int rc = 0;

	if (vtz_drvdata) {
		pr_err(PFX "vtz_driver_init(): device already initialized\n");
		return -EBUSY;
	}

	/* allocate private device memory, set to zero (kzalloc) !! */
	vtz_drvdata = kzalloc(sizeof(struct vtz_drvdata_t), GFP_KERNEL);
	if (!vtz_drvdata) {
		pr_err(PFX "vtz_driver_init(): kzalloc() failed\n");
		return -ENOMEM;
	}

	mutex_init(&vtz_drvdata->inside_tz_call_mutex);
	atomic_set(&vtz_drvdata->inside_short_tz_call_counter, 0);
	atomic_set(&vtz_drvdata->inside_long_tz_call_counter, 0);
	smp_mb__after_atomic();

	BUILD_BUG_ON((BUFFER_SIZE & (BUFFER_SIZE - 1))); //Assure that the buffer size is a power of 2
	mutex_init(&vtz_drvdata->log_buffer_mutex);
	init_waitqueue_head(&vtz_drvdata->log_event);
	vtz_buffer_clear();

	/* allocate device id range for all devices */
	rc = alloc_chrdev_region(&vtz_drvdata->dev, 0, DEVICES_COUNT, DRV_NAME);
	if (rc) {
		pr_err(PFX "vtz_driver_init(): alloc_chrdev_region() failed\n");
		vtz_drvdata->dev = 0;
		vtz_driver_exit();
		return rc;
	}

	/* create device class */
	vtz_drvdata->class = class_create(THIS_MODULE, DRV_NAME);
	if (IS_ERR(vtz_drvdata->class)) {
		pr_err(PFX "vtz_driver_init(): class_create() failed\n");
		rc = PTR_ERR(vtz_drvdata->class);
		vtz_drvdata->class = NULL;
		vtz_driver_exit();
		return rc;
	}

#ifdef CONFIG_VTZ2_STORAGE
	/* create device /dev/vtzstorage */
	rc = vtz_device_construct(
		&vtz_drvdata->dev_vdstorage,
		vtz_drvdata->class,
		"vtzstorage",
		&vtz_fops_storage,
		MKDEV(MAJOR(vtz_drvdata->dev), 0));
	if (rc) {
		vtz_driver_exit();
		return rc;
	}
#endif

#ifdef CONFIG_VTZ2_RSA
	/* create device /dev/vtzacrypt */
	rc = vtz_device_construct(
		&vtz_drvdata->dev_vdacrypt,
		vtz_drvdata->class,
		"vtzacrypt",
		&vtz_fops_acrypt,
		MKDEV(MAJOR(vtz_drvdata->dev), 1));
	if (rc) {
		vtz_driver_exit();
		return rc;
	}
#endif

#ifdef CONFIG_VTZ2_LOG
	/* create device /dev/vtzlog */
	rc = vtz_device_construct(
		&vtz_drvdata->dev_vdlog,
		vtz_drvdata->class,
		"vtzlog",
		&vtz_fops_log,
		MKDEV(MAJOR(vtz_drvdata->dev), 2));
	if (rc) {
		vtz_driver_exit();
		return rc;
	}
#endif

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_init(&vtz_drvdata->vtz_wakelock, WAKE_LOCK_IDLE, "vtz2");
#endif

	pr_info(PFX "initialized\n");

	return 0;
}

 

/******************/
/* module control */
/******************/

static int __init vtz_module_init(void)
{
	return vtz_driver_init();
}

static void __exit vtz_module_exit(void)
{
	vtz_driver_exit();
}

module_init(vtz_module_init);
module_exit(vtz_module_exit);
