/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */

#include <linux/slab.h>		/* kzalloc() */

#include <vtz2_mem.h>

void vtz_mem_set(vtz_memory_t *block, void* addr, size_t size)
{
	block->addr = addr;
	block->size = size;
}

void* vtz_mem_begin(vtz_memory_t *block)
{
	return block->addr;
}

void* vtz_mem_end(vtz_memory_t *block)
{
	uint8_t *ptr = (uint8_t*) block->addr;
	return &ptr[block->size];
}

size_t vtz_mem_size_adjust(size_t size, size_t alignment)
{	/* for block size aligning */
	return ((size+alignment-1)&(~(alignment-1)));
}

uint8_t* vtz_mem_page_adjust(uint8_t* addr, size_t alignment)
{	/* for block address aligning */
	return (uint8_t*)(((uintptr_t)&addr[alignment-1])&(~(alignment-1)));
}


/* blocks greater than 2048 */
/* buffer address has to be 4096 aligned */
/* buffer size has to be 4096 aligned */

void vtz_mem_check_alignment(vtz_memory_t *block, const char* block_name)
{
	/* pr_info(PFX "block '%s': address=%p, size=%zu\n", block_name, block->addr, block->size); */
	if(block->size > VTZ_MEM_BLOCK_LIMIT)			/* is block size more than 2k? */
	{
		if((((unsigned long)block->addr) & (VTZ_MEM_ALIGN_PAGE-1)) != 0 )	/* is block address aligned to 4k? */
		{
			pr_err(PFX "block '%s' with size %zu is unaligned\n", block_name, block->size);
		}
	}
}

int vtz_mem_malloc(	/* returns 0 if succeeded */
	vtz_mem_internal_t	*memory_layout,
	void			**request_addr,
	size_t			request_size,
	void			**response_addr,
	size_t			response_size,
	size_t			keyblob_size,	/* can be 0 */
	size_t			block_1_size,	/* can be 0, might be > 2048 */
	size_t			block_2_size)	/* can be 0, might be > 2048 */
{
	size_t	offset_request, offset_response, offset_keyblob, offset_block_1, offset_block_2, complete_size;
	uint8_t *memory;
	bool page_align = (block_1_size > VTZ_MEM_BLOCK_LIMIT) || (block_2_size > VTZ_MEM_BLOCK_LIMIT); 

	if (!memory_layout)
		return -1;
	
	offset_request		= 0;
	offset_response		= offset_request  + vtz_mem_size_adjust(request_size,  VTZ_MEM_ALIGN_NEON);
	offset_keyblob		= offset_response + vtz_mem_size_adjust(response_size, VTZ_MEM_ALIGN_NEON);
	offset_block_1		= offset_keyblob  + vtz_mem_size_adjust(keyblob_size,  VTZ_MEM_ALIGN_NEON);
	offset_block_2		= offset_block_1  + vtz_mem_size_adjust(block_1_size,  page_align ? VTZ_MEM_ALIGN_PAGE : VTZ_MEM_ALIGN_NEON); /* ensure size alignment */
	complete_size		= offset_block_2  + vtz_mem_size_adjust(block_2_size,  page_align ? VTZ_MEM_ALIGN_PAGE : VTZ_MEM_ALIGN_NEON); /* ensure size alignment */

	if(page_align)
		complete_size += VTZ_MEM_ALIGN_PAGE;			/* size reserve for page alignment */

	memory = (uint8_t*) kzalloc(complete_size, GFP_KERNEL);		/* always aligned at 8 bytes minimums */

	if (!memory)
		return -1;

	vtz_mem_set(&memory_layout->complete,	&memory[0],			complete_size);
	vtz_mem_set(&memory_layout->request,	&memory[offset_request],	request_size);
	vtz_mem_set(&memory_layout->response,	&memory[offset_response],	response_size);
	vtz_mem_set(&memory_layout->keyblob,	&memory[offset_keyblob],	keyblob_size);

	if(page_align)
	{
		vtz_mem_set(&memory_layout->block1,	vtz_mem_page_adjust(&memory[offset_block_1], VTZ_MEM_ALIGN_PAGE),	block_1_size);
		vtz_mem_set(&memory_layout->block2,	vtz_mem_page_adjust(&memory[offset_block_2], VTZ_MEM_ALIGN_PAGE),	block_2_size);
	} else {
		vtz_mem_set(&memory_layout->block1,	&memory[offset_block_1],	block_1_size);
		vtz_mem_set(&memory_layout->block2,	&memory[offset_block_2],	block_2_size);
	}

	/* pr_info(PFX "block 'complete': address=%p, size=%zu\n", memory_layout->complete.addr, memory_layout->complete.size); */

	vtz_mem_check_alignment(&memory_layout->request,  "request");
	vtz_mem_check_alignment(&memory_layout->response, "response");
	vtz_mem_check_alignment(&memory_layout->keyblob,  "keyblob");
	vtz_mem_check_alignment(&memory_layout->block1,   "block1");
	vtz_mem_check_alignment(&memory_layout->block2,   "block2");

	if (request_addr)
		*request_addr  = memory_layout->request.addr;
	if (response_addr)
		*response_addr = memory_layout->response.addr;

	return 0;
}

void vtz_mem_free(vtz_mem_internal_t *memory_layout)
{
	if (memory_layout && memory_layout->complete.addr) {
		/* cleanup temporary trustzone memory to prevent leaks */
		memset(memory_layout->complete.addr, 0, memory_layout->complete.size);
		kfree(memory_layout->complete.addr);
	}
}

int vtz_mem_copy_from_user(	/* returns 0 if succeeded */
	vtz_memory_t *kernel,
	vtz_memory_t *user)
{
	if (user->size > kernel->size)
		return -1;		/* prevent overflow */
	kernel->size = user->size;	/* set new kernelspace data size */
	return copy_from_user(kernel->addr, (void __user *)user->addr, user->size);
}

int vtz_mem_copy_to_user(	/* returns 0 if succeeded */
	vtz_memory_t *user,
	vtz_memory_t *kernel)
{
	if (kernel->size > user->size)
		return -1;		/* prevent overflow */
	user->size = kernel->size;	/* set new userspace data size */
	return copy_to_user((void __user *)user->addr, kernel->addr, kernel->size);
}

