#ifndef __VTZ2_MEM_H__
#define __VTZ2_MEM_H__
/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */


#include <linux/uaccess.h>	/* copy_from_user(), copy_to_user() */
#include <linux/string.h>	/* memset() */

#include <vtz_user.h> 		/* vtz user interface */
#include <vtz2_main.h>

/***************************/
/* vtz internal memory api */
/***************************/

#define VTZ_MEM_ALIGN_NEON 8
#define VTZ_MEM_ALIGN_PAGE 4096
#define VTZ_MEM_BLOCK_LIMIT 2048

static inline int vtz_nulltest(void * const ptr)	/* helper to prevent forbidden preprocessor warning */
{
	return (ptr==NULL);
}

#define TZ_MEM_IOCTL_INIT(KERNEL_PTR, USER_PTR)		\
	((sizeof(*KERNEL_PTR) != sizeof(*USER_PTR)) ||	\
	vtz_nulltest(KERNEL_PTR) ||			\
	vtz_nulltest(USER_PTR) ||			\
	(copy_from_user(KERNEL_PTR, USER_PTR, sizeof(*KERNEL_PTR)) != 0))

#define TZ_MEM_IOCTL_EXIT(KERNEL_PTR, USER_PTR)		\
	((sizeof(*KERNEL_PTR) != sizeof(*USER_PTR)) ||	\
	vtz_nulltest(KERNEL_PTR) ||			\
	vtz_nulltest(USER_PTR) ||			\
	(copy_to_user(USER_PTR, KERNEL_PTR, sizeof(*KERNEL_PTR)) != 0))


typedef struct {
	vtz_memory_t	complete;	/* complete allocated memory in one piece */
	vtz_memory_t	request;	/* section: request struct */
	vtz_memory_t	response;	/* section: response struct */
	vtz_memory_t	keyblob;	/* section: storage or RSA keyblob */
	vtz_memory_t	block1;		/* section: optional block (mostly input data) */
	vtz_memory_t	block2;		/* section: optional block (mostly output data) */
} vtz_mem_internal_t;

void vtz_mem_set(vtz_memory_t *block, void* addr, size_t size);
void* vtz_mem_begin(vtz_memory_t *block);
void* vtz_mem_end(vtz_memory_t *block);
size_t vtz_mem_size_adjust(size_t size, size_t alignment);
uint8_t* vtz_mem_page_adjust(uint8_t* addr, size_t alignment);

void vtz_mem_check_alignment(vtz_memory_t *block, const char* block_name);

int vtz_mem_malloc(vtz_mem_internal_t *memory_layout, void **request_addr, size_t request_size, void **response_addr, size_t response_size, size_t keyblob_size, size_t block_1_size, size_t block_2_size);

void vtz_mem_free(vtz_mem_internal_t *memory_layout);

int vtz_mem_copy_from_user( vtz_memory_t *kernel, vtz_memory_t *user); /* returns 0 if succeeded */
int vtz_mem_copy_to_user(vtz_memory_t *user, vtz_memory_t *kernel); /* returns 0 if succeeded */

#endif