/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */

#include <linux/errno.h>
#include <linux/fs.h>		 /*file ops*/

#include <vtz2_rsa.h>
#include <vtz_user.h>
#include <vtz2_main.h>
#include <vtz2_mem.h>
#include <vtz2_helper.h>
#include <vtz2_scm.h>

/*********************************************/
/* ioctl functions for registering a device  */
/*********************************************/

static int vtz_register(						/* returns 0 if succeeded */
	struct vtz_device_t		*private_device_data,
	vtz_ioctl_register_t __user	*ioctl_data_user)
{
	vtz_ioctl_register_t		ioctl_data_kernel;		/* stores kernel copy of ioctl user argument */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */

	if(private_device_data->transaction_lock == 0)			/* default when run first: 0 */
	{
		private_device_data->transaction_lock = (ioctl_data_kernel.transaction_challenge != 0);
	}
	else								/* device had already been opened and is locked */
	{								/* test transaction code */
//#ifndef PIN_MANAGEMENT_DISABLED
//		if(private_device_data->transaction != ioctl_data_kernel.transaction_challenge)
//			return vtz_throw_exception(-EPERM, NULL, TRANSAC_ERR,__func__, __LINE__);	/* device is locked and transaction is wrong */
//#endif
	}

	vtz_next_transaction(&private_device_data->transaction);	/* get transaction challenge */
	private_device_data->transaction |= 1;				/* set lowest bit for challenge multiplication*/
	ioctl_data_kernel.transaction_challenge = private_device_data->transaction;
	private_device_data->transaction *= private_device_data->pin;	/* calculate challenge */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, NULL,MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	return 0;
}


/**********************************/
/* ioctl functions for RSA crypto */
/**********************************/

static int vtz_rsa_generate(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_generate_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_generate_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_gen_keypair_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_gen_keypair_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			sizeof(crypto_rsa_key_type_t),			/* keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	request->cmd_id = CRYPTO_STORAGE_GENERATE_KEY_OLD;		/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->rsa_params.modulus_size = ioctl_data_kernel.modulus_bitlen;
	request->rsa_params.public_exponent = ioctl_data_kernel.public_exponent;
	request->rsa_params.digest_pad_type = ioctl_data_kernel.digest_pad_type;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_GENERATE_KEY_OLD)	/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.keyblob.size = response->key_blob_size;		/* get response data, adjust keyblob size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT ,&tz_memory, MEM_COPY_FROM_TZ_ERR, __func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}


static int vtz_rsa_sign(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_sign_data_cmd_t		*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_sign_data_resp_t		*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.data.size,			/* data storage size */
			ioctl_data_kernel.signature.size))		/* signature storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_SIGN_DATA_OLD;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->dlen = tz_memory.block1.size;
	request->signeddata = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->signeddata_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_SIGN_DATA_OLD)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sig_len;			/* get response data, adjust signature size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.signature, &tz_memory.block2))	/* copy kernel signature into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_verify(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_verify_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_asym_verify_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.data.size,			/* data storage size */
			ioctl_data_kernel.signature.size))		/* signature storage size */
		return vtz_throw_exception(-ENOBUFS,NULL, MEM_ALLOC_ERR, __func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory,  MEM_COPY_TO_TZ_ERR,__func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, MEM_COPY_TO_TZ_ERR, __func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block2, &ioctl_data_kernel.signature))	/* copy userspace signature into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, MEM_COPY_TO_TZ_ERR, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_VERIFY_DATA_OLD;		/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->signed_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->signed_dlen = tz_memory.block1.size;
	request->signature = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->slen = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_VERIFY_DATA_OLD)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_export(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_export_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_export_t			ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_storage_rsa_export_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_storage_rsa_export_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */
	uint8_t* public_exponent_ptr;
	uint32_t i;

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.modulus.size,			/* modulus storage size */
			sizeof(ioctl_data_kernel.public_exponent)))	/* public exponent storage size */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory,MEM_COPY_TO_TZ_ERR, __func__, __LINE__);

	request->cmd_id = CRYPTO_STORAGE_EXPORT_PUBKEY;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->export_format = CRYPTO_STORAGE_FORMAT_RAW_BYTES;
	request->modulus = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
	request->modulus_size = tz_memory.block1.size;
	request->public_exponent = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
	request->public_exponent_size = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR, __func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_EXPORT_PUBKEY)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR, __func__, __LINE__);	/* positive return value! */

	tz_memory.block1.size = response->modulus_size;
	if (vtz_mem_copy_to_user(&ioctl_data_kernel.modulus, &tz_memory.block1))	/* copy kernel modulus into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	if(response->public_exponent_size > sizeof(ioctl_data_kernel.public_exponent))
		return vtz_throw_exception(-EOVERFLOW, &tz_memory, RSA_PUBLIC_EXP_OF_ERR,__func__, __LINE__);

	ioctl_data_kernel.public_exponent = 0;
	public_exponent_ptr = (uint8_t*)tz_memory.block2.addr;
	for(i=response->public_exponent_size; i>0; i--)
		ioctl_data_kernel.public_exponent = (ioctl_data_kernel.public_exponent << 8) | (uint64_t)(*public_exponent_ptr++);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_sign_mhash(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_sign_verify_t __user	*ioctl_data_user)
{
	vtz_ioctl_rsa_sign_verify_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_asym_sign_data_with_pad_cmd_t	*request;		/* trustzone request (inside tz_memory) */
#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	tz_crypto_asym_sign_data_cmd_t		*request_wa;		/* enable workaround for SHA2-256, */
	bool enable_workaround;						/* if CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD may be unavailable */
#endif
	tz_crypto_asym_sign_data_resp_t		*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */

#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	enable_workaround = (ioctl_data_kernel.data.size==SHA256_HASH_LENGTH);
#endif


#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	if(enable_workaround)
	{
		if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
				(void**)&request_wa, sizeof(*request_wa),	/* adapted request_wa struct */
				(void**)&response, sizeof(*response),		/* adapted response struct */
				ioctl_data_kernel.keyblob.size,			/* keyblob size */
				ioctl_data_kernel.data.size,			/* hash size */
				ioctl_data_kernel.signature.size))		/* signature storage size */
			return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */
	}
	else
	{
#endif
		if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
				(void**)&request, sizeof(*request),		/* adapted request struct */
				(void**)&response, sizeof(*response),		/* adapted response struct */
				ioctl_data_kernel.keyblob.size,			/* keyblob size */
				ioctl_data_kernel.data.size,			/* hash size */
				ioctl_data_kernel.signature.size))		/* signature storage size */
			return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */
#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	}
#endif

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.data))		/* copy userspace hash into kernel */
		return vtz_throw_exception(-EFAULT,&tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);

#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	if(enable_workaround)
	{
		request_wa->cmd_id = CRYPTO_STORAGE_SIGN_MHASH;				/* set request_wa data */
		request_wa->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
		request_wa->key_blob.key_material_len = tz_memory.keyblob.size;
		request_wa->data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
		request_wa->dlen = tz_memory.block1.size;
		request_wa->signeddata = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
		request_wa->signeddata_len = tz_memory.block2.size;
	}
	else
	{
#endif
		request->cmd_id = CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD;		/* set request data */
		request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
		request->key_blob.key_material_len = tz_memory.keyblob.size;
		request->data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);;
		request->dlen = tz_memory.block1.size;
		request->signeddata = (uint8_t*)virt_to_phys(tz_memory.block2.addr);;
		request->signeddata_len = tz_memory.block2.size;

		switch(tz_memory.block1.size)					/* identify padding type by hash buffer size */
		{
			case SHA256_HASH_LENGTH:
				request->digest_pad_type = TZ_RSA_PKCS115_SHA2_256;
				break;
			case SHA384_HASH_LENGTH:
				request->digest_pad_type = TZ_RSA_PKCS115_SHA2_384;
				break;
			case SHA512_HASH_LENGTH:
				request->digest_pad_type = TZ_RSA_PKCS115_SHA2_512;
				break;
			default:
				return vtz_throw_exception(-EMSGSIZE, &tz_memory, RSA_HASH_SIZE_ERR,__func__, __LINE__);
		}
#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	}
#endif

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	if(enable_workaround)
	{
		if (response->cmd_id != CRYPTO_STORAGE_SIGN_MHASH)	/* wrong response id */
			return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);
	}
	else
	{
#endif
		if (response->cmd_id != CRYPTO_STORAGE_SIGN_MHASH_WITH_PAD)	/* wrong response id */
			return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);
#ifdef WORKAROUND_MISSING_SIGN_MHASH_WITH_PAD
	}
#endif

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sig_len;			/* get response data, adjust signature size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.signature, &tz_memory.block2))	/* copy kernel signature into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_rsa_import(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_rsa_import_t __user		*ioctl_data_user)
{
	vtz_ioctl_rsa_import_t			ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_crypto_storage_rsa_import_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_crypto_storage_rsa_import_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */
	uint32_t i;

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if((ioctl_data_kernel.public_exponent > 0xFFFFFFFFFFLL) ||	/* check for input buffer overflows & wrong arguments */
			(ioctl_data_kernel.public_exponent == 0) ||
			(ioctl_data_kernel.modulus.size > CRYPTO_ASYM_KEY_SIZE_MAX) ||
			(ioctl_data_kernel.private_exponent.size > CRYPTO_ASYM_KEY_SIZE_MAX))
		return vtz_throw_exception(-EOVERFLOW, NULL, RSA_PUBLIC_EXP_OF_ERR,__func__, __LINE__);	/* error with input buffers */

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	request->cmd_id = CRYPTO_STORAGE_IMPORT_KEY;			/* set request data */
	request->key_blob.key_material = (crypto_rsa_key_type_t*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;

	/* copy modulus and private exponent from user mode to tz struct */
	if(copy_from_user(&request->modulus[0], ioctl_data_kernel.modulus.addr, ioctl_data_kernel.modulus.size))
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);
	request->modulus_size = ioctl_data_kernel.modulus.size;
	if(copy_from_user(&request->private_exponent[0], ioctl_data_kernel.private_exponent.addr, ioctl_data_kernel.private_exponent.size))
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);
	request->private_exponent_size = ioctl_data_kernel.private_exponent.size;

	/* copy public exponent to tz struct (force big endian) */
	for(i = CRYPTO_ASYM_PUB_EXP_SIZE_MAX; i > 0; i--)		/* 5..1, search highest byte != 0 */
		if((ioctl_data_kernel.public_exponent & (0xFFLL << (8*(i-1)))) != 0) /* 0xFF00000000, 0xFF000000, 0xFF0000, 0xFF00, 0xFF */
			break;
	request->public_exponent_size = i;
	for(; i > 0; i--)						/* x..1, copy only used bytes */
		request->public_exponent[request->public_exponent_size - i] = (uint8_t)(ioctl_data_kernel.public_exponent >> (8*(i-1)));

	request->digest_pad_type = ioctl_data_kernel.digest_pad_type;	/* set padding type */

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_RSA_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != CRYPTO_STORAGE_IMPORT_KEY)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT ,&tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

/*********************/
/* file i/o vdacrypt */
/*********************/

static int vtz_acrypt_file_open(struct inode *inode, struct file *file)
{
	struct vtz_drvdata_t *private_driver_data = vtz_drvdata;

	if (!private_driver_data)
		return -ENODATA;

	if (inode->i_rdev != private_driver_data->dev_vdacrypt.dev)
		return -ENXIO;

	if (file->private_data)		/* device already open? */
		return -EBADF;

	file->private_data = (void*) &private_driver_data->dev_vdacrypt;

	return nonseekable_open(inode, file);
}

static int vtz_acrypt_file_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}

static long vtz_acrypt_file_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct vtz_device_t *private_device_data = (struct vtz_device_t*) file->private_data;
	long rc;

	if(!private_device_data)
		return -ENODATA;

	if(cmd == VTZ_IOCTL_RSA_GENERATE) {
		vtz_enter_critical_long_tz_duration();
	} else {
		rc = vtz_enter_critical_short_tz_duration();
		if(rc)
			return rc;
	}

	switch (cmd) {
	case VTZ_IOCTL_RSA_REGISTER:
		rc = vtz_register(private_device_data, (vtz_ioctl_register_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_GENERATE:
		rc = vtz_rsa_generate(private_device_data, (vtz_ioctl_rsa_generate_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_SIGN:
		rc = vtz_rsa_sign(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_VERIFY:
		rc = vtz_rsa_verify(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_EXPORT:
		rc = vtz_rsa_export(private_device_data, (vtz_ioctl_rsa_export_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_SIGN_MHASH:
		rc = vtz_rsa_sign_mhash(private_device_data, (vtz_ioctl_rsa_sign_verify_t __user *)arg);
		break;
	case VTZ_IOCTL_RSA_IMPORT:
		rc = vtz_rsa_import(private_device_data, (vtz_ioctl_rsa_import_t __user *)arg);
		break;
	default:
		rc = -ENOTTY;
		break;
	}

	if(cmd == VTZ_IOCTL_RSA_GENERATE)
		vtz_leave_critical_long_tz_duration();
	else
		vtz_leave_critical_short_tz_duration();
	return rc;
}

const struct file_operations vtz_fops_acrypt = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl	= vtz_acrypt_file_ioctl,
	.open		= vtz_acrypt_file_open,
	.release	= vtz_acrypt_file_release,
	.llseek		= no_llseek,
};
