/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */

#include <linux/errno.h>
#include <linux/fs.h>		 /*file ops*/

#include <vtz2_storage.h>
#include <vtz_user.h>
#include <vtz2_main.h>
#include <vtz2_mem.h>
#include <vtz2_helper.h>
#include <vtz2_scm.h>

/*********************************************/
/* ioctl functions for registering a device  */
/*********************************************/

static int vtz_register(						/* returns 0 if succeeded */
	struct vtz_device_t		*private_device_data,
	vtz_ioctl_register_t __user	*ioctl_data_user)
{
	vtz_ioctl_register_t		ioctl_data_kernel;		/* stores kernel copy of ioctl user argument */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */

	if(private_device_data->transaction_lock == 0)			/* default when run first: 0 */
	{
		private_device_data->transaction_lock = (ioctl_data_kernel.transaction_challenge != 0);
	}
	else								/* device had already been opened and is locked */
	{								/* test transaction code */
//#ifndef PIN_MANAGEMENT_DISABLED
//		if(private_device_data->transaction != ioctl_data_kernel.transaction_challenge)
//			return vtz_throw_exception(-EPERM, NULL, TRANSAC_ERR,__func__, __LINE__);	/* device is locked and transaction is wrong */
//#endif
	}

	vtz_next_transaction(&private_device_data->transaction);	/* get transaction challenge */
	private_device_data->transaction |= 1;				/* set lowest bit for challenge multiplication*/
	ioctl_data_kernel.transaction_challenge = private_device_data->transaction;
	private_device_data->transaction *= private_device_data->pin;	/* calculate challenge */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	return 0;
}

/**************************************/
/* ioctl functions for secure storage */
/**************************************/

static int vtz_storage_generate(					/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_generate_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_generate_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_gen_key_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_gen_key_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			VTZ_STORAGE_KEYBLOB_SIZE,			/* maximum keyblob size */
			0, 0))						/* not used here */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	request->cmd_id = TZ_STORAGE_SERVICE_GENERATE_KEY;		/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_GENERATE_KEY)	/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.keyblob.size = response->key_blob_size;		/* get response data, adjust keyblob size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.keyblob, &tz_memory.keyblob))	/* copy kernel keyblob into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT,&tz_memory, MEM_COPY_TO_USR_ERR, __func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_storage_seal(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_seal_unseal_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_seal_unseal_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_seal_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_seal_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */

	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.unsealed.size,		/* uncrypted data buffer */
			ioctl_data_kernel.sealed.size))			/* crypted data buffer */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.unsealed))	/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);

	request->cmd_id = TZ_STORAGE_SERVICE_SEAL_DATA;			/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->plain_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);
	request->plain_dlen = tz_memory.block1.size;
	request->output_buffer = (uint8_t*)virt_to_phys(tz_memory.block2.addr);
	request->output_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_SEAL_DATA)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->sealed_data_len;		/* get response data, adjust sealed data size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.sealed, &tz_memory.block2))	/* copy kernel sealed data into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}

static int vtz_storage_unseal(						/* returns 0 if succeeded */
	struct vtz_device_t			*private_device_data,
	vtz_ioctl_storage_seal_unseal_t __user	*ioctl_data_user)
{
	vtz_ioctl_storage_seal_unseal_t		ioctl_data_kernel;	/* stores kernel copy of ioctl user argument */
	vtz_mem_internal_t			tz_memory;		/* stores all allocated memory */
	tz_storage_service_unseal_data_cmd_t	*request;		/* trustzone request (inside tz_memory) */
	tz_storage_service_unseal_data_resp_t	*response;		/* trustzone response (inside tz_memory) */
	int rc;								/* secure monitor call return code */

	if (TZ_MEM_IOCTL_INIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument into kernel space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);	/* error with user memory */


	if (vtz_mem_malloc(&tz_memory,					/* allocate kernel memory for call */
			(void**)&request, sizeof(*request),		/* adapted request struct */
			(void**)&response, sizeof(*response),		/* adapted response struct */
			ioctl_data_kernel.keyblob.size,			/* keyblob size */
			ioctl_data_kernel.sealed.size,			/* crypted data buffer */
			ioctl_data_kernel.unsealed.size))		/* uncrypted data buffer */
		return vtz_throw_exception(-ENOBUFS, NULL, MEM_ALLOC_ERR,__func__, __LINE__);	/* No buffer space available */

	if (vtz_mem_copy_from_user(&tz_memory.keyblob, &ioctl_data_kernel.keyblob))	/* copy userspace keyblob into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);
	if (vtz_mem_copy_from_user(&tz_memory.block1, &ioctl_data_kernel.sealed))	/* copy userspace data into kernel */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_TZ_ERR,__func__, __LINE__);

	request->cmd_id = TZ_STORAGE_SERVICE_UNSEAL_DATA;		/* set request data */
	request->key_blob.key_material = (void*)virt_to_phys(tz_memory.keyblob.addr);
	request->key_blob.key_material_len = tz_memory.keyblob.size;
	request->sealed_data = (uint8_t*)virt_to_phys(tz_memory.block1.addr);
	request->sealed_dlen = tz_memory.block1.size;
	request->output_buffer = (uint8_t*)virt_to_phys(tz_memory.block2.addr);
	request->output_len = tz_memory.block2.size;

	rc = vtz_scm_call(TZ_SVC_CRYPTO, TZ_SVC_SYM_ID, &tz_memory);	/* call trustzone */

	if (rc!=0)							/* error in internal scm_call */
		return vtz_throw_exception(rc, &tz_memory, SCM_ERR,__func__, __LINE__);	/* individual error code */

	if (response->cmd_id != TZ_STORAGE_SERVICE_UNSEAL_DATA)		/* wrong response id */
		return vtz_throw_exception(-EINVAL, &tz_memory, RESP_ERR,__func__, __LINE__);

	if (response->status != 0)					/* error in trustzone call */
		return vtz_throw_exception(response->status, &tz_memory, STAT_ERR,__func__, __LINE__);	/* positive return value! */

	tz_memory.block2.size = response->unsealed_data_len;		/* get response data, adjust unsealed data size */

	if (vtz_mem_copy_to_user(&ioctl_data_kernel.unsealed, &tz_memory.block2))	/* copy kernel unsealed data into userspace */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_FROM_TZ_ERR,__func__, __LINE__);

	vtz_next_transaction(&ioctl_data_kernel.transaction);		/* calc next transaction id */

	if (TZ_MEM_IOCTL_EXIT(&ioctl_data_kernel, ioctl_data_user))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, &tz_memory, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	private_device_data->transaction = ioctl_data_kernel.transaction;	/* update transaction */

	vtz_mem_free(&tz_memory);					/* free kernel memory */
	return 0;
}



/**********************/
/* file i/o vdstorage */
/**********************/

static int vtz_storage_file_open(struct inode *inode, struct file *file)
{
	struct vtz_drvdata_t *private_driver_data = vtz_drvdata;

	if (!private_driver_data)
		return -ENODATA;

	if (inode->i_rdev != private_driver_data->dev_vdstorage.dev)
		return -ENXIO;

	if (file->private_data)		/* device already open? */
		return -EBADF;

	file->private_data = (void*) &private_driver_data->dev_vdstorage;

	return nonseekable_open(inode, file);
}

static int vtz_storage_file_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}

static long vtz_storage_file_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct vtz_device_t *private_device_data = (struct vtz_device_t*) file->private_data;
	long rc;

	if(!private_device_data)
		return -ENODATA;

	rc = vtz_enter_critical_short_tz_duration();
	if(rc)
		return rc;

	switch (cmd) {
	case VTZ_IOCTL_STORAGE_REGISTER:
		rc = vtz_register(private_device_data, (vtz_ioctl_register_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_GENERATE:
		rc = vtz_storage_generate(private_device_data, (vtz_ioctl_storage_generate_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_SEAL:
		rc = vtz_storage_seal(private_device_data, (vtz_ioctl_storage_seal_unseal_t __user *)arg);
		break;
	case VTZ_IOCTL_STORAGE_UNSEAL:
		rc = vtz_storage_unseal(private_device_data, (vtz_ioctl_storage_seal_unseal_t __user *)arg);
		break;
	default:
		rc = -ENOTTY;
		break;
	}

	vtz_leave_critical_short_tz_duration();
	return rc;
}

const struct file_operations vtz_fops_storage = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl	= vtz_storage_file_ioctl,
	.open		= vtz_storage_file_open,
	.release	= vtz_storage_file_release,
	.llseek		= no_llseek,
};


