/* Copyright (c) 2015-2018, 2020 The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#ifndef LINUX_MMC_CQ_HCI_H
#define LINUX_MMC_CQ_HCI_H
#include <linux/mmc/core.h>
#include <linux/platform_device.h>
#include <linux/keyslot-manager.h>

/* registers */
/* version */
#define CQVER		0x00
/* capabilities */
#define CQCAP		0x04
#define CQ_CAP_CS	(1 << 28)
#define CQ_CCAP		0x100
#define CQ_CRYPTOCAP	0x104
/* configuration */
#define CQCFG		0x08
#define CQ_DCMD		0x00001000
#define CQ_TASK_DESC_SZ 0x00000100
#define CQ_ENABLE	0x00000001
#define CQ_ICE_ENABLE	0x00000002

/* control */
#define CQCTL		0x0C
#define CLEAR_ALL_TASKS 0x00000100
#define HALT		0x00000001

/* interrupt status */
#define CQIS		0x10
#define CQIS_HAC	(1 << 0)
#define CQIS_TCC	(1 << 1)
#define CQIS_RED	(1 << 2)
#define CQIS_TCL	(1 << 3)
#define CQIS_GCE	(1 << 4)
#define CQIS_ICCE	(1 << 5)

/* interrupt status enable */
#define CQISTE		0x14

/* interrupt signal enable */
#define CQISGE		0x18

/* interrupt coalescing */
#define CQIC		0x1C
#define CQIC_ENABLE	(1 << 31)
#define CQIC_RESET	(1 << 16)
#define CQIC_ICCTHWEN	(1 << 15)
#define CQIC_ICCTH(x)	((x & 0x1F) << 8)
#define CQIC_ICTOVALWEN (1 << 7)
#define CQIC_ICTOVAL(x) (x & 0x7F)

/* task list base address */
#define CQTDLBA		0x20

/* task list base address upper */
#define CQTDLBAU	0x24

/* door-bell */
#define CQTDBR		0x28

/* task completion notification */
#define CQTCN		0x2C

/* device queue status */
#define CQDQS		0x30

/* device pending tasks */
#define CQDPT		0x34

/* task clear */
#define CQTCLR		0x38

/* send status config 1 */
#define CQSSC1		0x40
/*
 * Value n means CQE would send CMD13 during the transfer of data block
 * BLOCK_CNT-n
 */
#define SEND_QSR_INTERVAL 0x70001

/* send status config 2 */
#define CQSSC2		0x44

/* response for dcmd */
#define CQCRDCT		0x48

/* response mode error mask */
#define CQRMEM		0x50
#define CQ_EXCEPTION	(1 << 6)

/* task error info */
#define CQTERRI		0x54

/* CQTERRI bit fields */
#define CQ_RMECI	0x1F
#define CQ_RMETI	(0x1F << 8)
#define CQ_RMEFV	(1 << 15)
#define CQ_DTECI	(0x3F << 16)
#define CQ_DTETI	(0x1F << 24)
#define CQ_DTEFV	(1 << 31)

#define GET_CMD_ERR_TAG(__r__) ((__r__ & CQ_RMETI) >> 8)
#define GET_DAT_ERR_TAG(__r__) ((__r__ & CQ_DTETI) >> 24)

/* command response index */
#define CQCRI		0x58

/* command response argument */
#define CQCRA		0x5C

#define CQ_INT_ALL	0x3F
#define CQIC_DEFAULT_ICCTH 31
#define CQIC_DEFAULT_ICTOVAL 1

/* attribute fields */
#define VALID(x)	((x & 1) << 0)
#define END(x)		((x & 1) << 1)
#define INT(x)		((x & 1) << 2)
#define ACT(x)		((x & 0x7) << 3)

/* data command task descriptor fields */
#define FORCED_PROG(x)	((x & 1) << 6)
#define CONTEXT(x)	((x & 0xF) << 7)
#define DATA_TAG(x)	((x & 1) << 11)
#define DATA_DIR(x)	((x & 1) << 12)
#define PRIORITY(x)	((x & 1) << 13)
#define QBAR(x)		((x & 1) << 14)
#define REL_WRITE(x)	((x & 1) << 15)
#define BLK_COUNT(x)	((x & 0xFFFF) << 16)
#define BLK_ADDR(x)	((x & 0xFFFFFFFF) << 32)

/* direct command task descriptor fields */
#define CMD_INDEX(x)	((x & 0x3F) << 16)
#define CMD_TIMING(x)	((x & 1) << 22)
#define RESP_TYPE(x)	((x & 0x3) << 23)

/* transfer descriptor fields */
#define DAT_LENGTH(x)	((x & 0xFFFF) << 16)
#define DAT_ADDR_LO(x)	((x & 0xFFFFFFFF) << 32)
#define DAT_ADDR_HI(x)	((x & 0xFFFFFFFF) << 0)
#define DATA_UNIT_NUM(x)	(((u64)(x) & 0xFFFFFFFF) << 0)
#define CRYPTO_CONFIG_INDEX(x)	(((u64)(x) & 0xFF) << 32)
#define CRYPTO_ENABLE(x)	(((u64)(x) & 0x1) << 47)

/* ICE context is present in the upper 64bits of task descriptor */
#define CQ_TASK_DESC_ICE_PARAM_OFFSET	8
/* ICE descriptor size */
#define CQ_TASK_DESC_ICE_PARAMS_SIZE	8

/*
 * Add new macro for updated CQ vendor specific
 * register address for SDHC v5.0 onwards.
 */
#define CQ_V5_VENDOR_CFG	0x900
#define CQ_VENDOR_CFG	0x100
#define CMDQ_SEND_STATUS_TRIGGER (1 << 31)

struct cmdq_host;

/* CCAP - Crypto Capability 100h */
union cmdq_crypto_capabilities {
	__le32 reg_val;
	struct {
		u8 num_crypto_cap;
		u8 config_count;
		u8 reserved;
		u8 config_array_ptr;
	};
};

enum cmdq_crypto_key_size {
	CMDQ_CRYPTO_KEY_SIZE_INVALID	= 0x0,
	CMDQ_CRYPTO_KEY_SIZE_128	= 0x1,
	CMDQ_CRYPTO_KEY_SIZE_192	= 0x2,
	CMDQ_CRYPTO_KEY_SIZE_256	= 0x3,
	CMDQ_CRYPTO_KEY_SIZE_512	= 0x4,
};

enum cmdq_crypto_alg {
	CMDQ_CRYPTO_ALG_AES_XTS		= 0x0,
	CMDQ_CRYPTO_ALG_BITLOCKER_AES_CBC	= 0x1,
	CMDQ_CRYPTO_ALG_AES_ECB		= 0x2,
	CMDQ_CRYPTO_ALG_ESSIV_AES_CBC		= 0x3,
};

/* x-CRYPTOCAP - Crypto Capability X */
union cmdq_crypto_cap_entry {
	__le32 reg_val;
	struct {
		u8 algorithm_id;
		u8 sdus_mask; /* Supported data unit size mask */
		u8 key_size;
		u8 reserved;
	};
};

#define CMDQ_CRYPTO_CONFIGURATION_ENABLE (1 << 7)
#define CMDQ_CRYPTO_KEY_MAX_SIZE 64

/* x-CRYPTOCFG - Crypto Configuration X */
union cmdq_crypto_cfg_entry {
	__le32 reg_val[32];
	struct {
		u8 crypto_key[CMDQ_CRYPTO_KEY_MAX_SIZE];
		u8 data_unit_size;
		u8 crypto_cap_idx;
		u8 reserved_1;
		u8 config_enable;
		u8 reserved_multi_host;
		u8 reserved_2;
		u8 vsb[2];
		u8 reserved_3[56];
	};
};

struct cmdq_host_crypto_variant_ops {
	void (*setup_rq_keyslot_manager)(struct cmdq_host *host,
	struct request_queue *q);
	void (*destroy_rq_keyslot_manager)(struct cmdq_host *host,
					   struct request_queue *q);
#ifdef CONFIG_BLK_INLINE_ENCRYPTION
	int (*host_init_crypto)(struct cmdq_host *host,
				const struct keyslot_mgmt_ll_ops *ksm_ops);
#endif
	void (*enable)(struct cmdq_host *host);
	void (*disable)(struct cmdq_host *host);
	int (*suspend)(struct cmdq_host *host);
	int (*resume)(struct cmdq_host *host);
	int (*debug)(struct cmdq_host *host);
	int (*prepare_crypto_desc)(struct cmdq_host *host,
				   struct mmc_request *mrq, u64 *ice_ctx);
	int (*complete_crypto_desc)(struct cmdq_host *host,
				    struct mmc_request *mrq, u64 *ice_ctx);
	int (*reset)(struct cmdq_host *host);
	int (*recovery_finish)(struct cmdq_host *host);
	int (*program_key)(struct cmdq_host *host,
			   const union cmdq_crypto_cfg_entry *cfg, int slot);
	void *priv;
};

struct task_history {
	u64 task;
	bool is_dcmd;
};

struct cmdq_host {
	const struct cmdq_host_ops *ops;
	void __iomem *mmio;
	void __iomem *icemmio;
	struct mmc_host *mmc;

	/* 64 bit DMA */
	bool dma64;
	int num_slots;

	u32 dcmd_slot;
	u32 caps;
#define CMDQ_TASK_DESC_SZ_128 0x1
#define CMDQ_CAP_CRYPTO_SUPPORT 0x2

	u32 quirks;
#define CMDQ_QUIRK_SHORT_TXFR_DESC_SZ 0x1
#define CMDQ_QUIRK_NO_DCMD	0x2

	bool enabled;
	bool halted;
	bool init_done;
	bool offset_changed;

	u8 *desc_base;

	/* total descriptor size */
	u8 slot_sz;

	/* 64/128 bit depends on CQCFG */
	u8 task_desc_len;

	/* 64 bit on 32-bit arch, 128 bit on 64-bit */
	u8 link_desc_len;

	u8 *trans_desc_base;
	/* same length as transfer descriptor */
	u8 trans_desc_len;

	dma_addr_t desc_dma_base;
	dma_addr_t trans_desc_dma_base;

	struct task_history *thist;
	u8 thist_idx;

	struct completion halt_comp;
	struct mmc_request **mrq_slot;
	void *private;
	const struct cmdq_host_crypto_variant_ops *crypto_vops;
#ifdef CONFIG_MMC_CQ_HCI_CRYPTO
	union cmdq_crypto_capabilities crypto_capabilities;
	union cmdq_crypto_cap_entry *crypto_cap_array;
	u32 crypto_cfg_register;
#ifdef CONFIG_BLK_INLINE_ENCRYPTION
	struct keyslot_manager *ksm;
#endif /* CONFIG_BLK_INLINE_ENCRYPTION */
#endif /* CONFIG_SCSI_CQHCI_CRYPTO */
};

struct cmdq_host_ops {
	void (*set_transfer_params)(struct mmc_host *mmc);
	void (*set_data_timeout)(struct mmc_host *mmc, u32 val);
	void (*clear_set_irqs)(struct mmc_host *mmc, bool clear);
	void (*set_block_size)(struct mmc_host *mmc);
	void (*dump_vendor_regs)(struct mmc_host *mmc);
	void (*write_l)(struct cmdq_host *host, u32 val, int reg);
	u32 (*read_l)(struct cmdq_host *host, int reg);
	void (*clear_set_dumpregs)(struct mmc_host *mmc, bool set);
	void (*enhanced_strobe_mask)(struct mmc_host *mmc, bool set);
	int (*reset)(struct mmc_host *mmc);
	void (*post_cqe_halt)(struct mmc_host *mmc);
};

static inline void cmdq_writel(struct cmdq_host *host, u32 val, int reg)
{
	if (unlikely(host->ops && host->ops->write_l))
		host->ops->write_l(host, val, reg);
	else
		writel_relaxed(val, host->mmio + reg);
}

static inline u32 cmdq_readl(struct cmdq_host *host, int reg)
{
	if (unlikely(host->ops && host->ops->read_l))
		return host->ops->read_l(host, reg);
	else
		return readl_relaxed(host->mmio + reg);
}

extern irqreturn_t cmdq_irq(struct mmc_host *mmc, int err);
extern int cmdq_init(struct cmdq_host *cq_host, struct mmc_host *mmc,
		     bool dma64);
extern struct cmdq_host *cmdq_pltfm_init(struct platform_device *pdev);
#endif
