/*
 *  Copyright (c) 2018-2021, The Linux Foundation. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *    * Neither the name of The Linux Foundation nor the names of its
 *      contributors may be used to endorse or promote products derived
 *      from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 *  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *  OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 *  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @file: Cv2xTxApp.cpp
 *
 * @brief: Simple application that demonstrates Tx in Cv2x
 */

#include <assert.h>
#include <ifaddrs.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include <iostream>
#include <memory>

#include <telux/cv2x/Cv2xRadio.hpp>
#include <telux/cv2x/Cv2xUtil.hpp>

#include "../../../common/utils/Utils.hpp"

using std::array;
using std::cerr;
using std::cout;
using std::endl;
using std::promise;
using std::shared_ptr;
using telux::common::ErrorCode;
using telux::common::Status;
using telux::common::ServiceStatus;
using telux::cv2x::Cv2xFactory;
using telux::cv2x::Cv2xStatus;
using telux::cv2x::Cv2xStatusType;
using telux::cv2x::ICv2xTxFlow;
using telux::cv2x::Periodicity;
using telux::cv2x::Priority;
using telux::cv2x::TrafficCategory;
using telux::cv2x::TrafficIpType;
using telux::cv2x::EventFlowInfo;
using telux::cv2x::SpsFlowInfo;
using telux::cv2x::DataSessionSettings;
using telux::cv2x::Cv2xUtil;

static constexpr uint32_t TX_SERVICE_ID = 1u;
static constexpr uint16_t TX_SRC_PORT_NUM = 2500u;
static constexpr uint32_t G_BUF_LEN = 128;
static constexpr uint16_t NUM_TEST_ITERATIONS = 128;
static constexpr int      PRIORITY = 3;

static constexpr char TEST_VERNO_MAGIC = 'Q';
static constexpr char UEID = 1;

enum class TxFlowType {
    SpsOnly,
    EventOnly,
};

static std::shared_ptr<telux::cv2x::ICv2xRadio> gCv2xRadio;
static Cv2xStatus gCv2xStatus;
static promise<ErrorCode> gCallbackPromise;
static shared_ptr<ICv2xTxFlow> gTxFlow;
static array<char, G_BUF_LEN> gBuf;
static TxFlowType gFlowType = TxFlowType::SpsOnly;
static bool gAutoRetransMode = true;

// Resets the global callback promise
static inline void resetCallbackPromise(void) {
    gCallbackPromise = promise<ErrorCode>();
}

// Callback function for ICv2xRadioManager->requestCv2xStatus()
static void cv2xStatusCallback(Cv2xStatus status, ErrorCode error) {
    if (ErrorCode::SUCCESS == error) {
        gCv2xStatus = status;
    }
    gCallbackPromise.set_value(error);
}

// Callback function for ICv2xRadio->createTxSpsFlow()
static void createSpsFlowCallback(shared_ptr<ICv2xTxFlow> txSpsFlow,
                                  shared_ptr<ICv2xTxFlow> unusedFlow,
                                  ErrorCode spsError,
                                  ErrorCode unusedError) {
    if (ErrorCode::SUCCESS == spsError) {
        gTxFlow = txSpsFlow;
    }
    gCallbackPromise.set_value(spsError);
}

// Callback function for ICv2xRadio->createTxEventFlow()
static void createEventFlowCallback(shared_ptr<ICv2xTxFlow> txEventFlow,
                                  ErrorCode eventError) {
    if (ErrorCode::SUCCESS == eventError) {
        gTxFlow = txEventFlow;
    }
    gCallbackPromise.set_value(eventError);
}

// Callback function for ICv2xRadio->changeEventFlowInfo()
static void changeEventFlowInfoCallback(shared_ptr<ICv2xTxFlow> txEventFlow,
                                  ErrorCode eventError) {
    if (ErrorCode::SUCCESS == eventError) {
        gTxFlow = txEventFlow;
    }
    gCallbackPromise.set_value(eventError);
}

// Callback function for ICv2xRadio->changeSpsFlowInfo()
static void changeSpsFlowInfoCallback(shared_ptr<ICv2xTxFlow> txSpsFlow,
                                  ErrorCode spsError) {
    if (ErrorCode::SUCCESS == spsError) {
        gTxFlow = txSpsFlow;
    }
    gCallbackPromise.set_value(spsError);
}

// Callback function for ICv2xRadio->requestSpsFlowInfo()
static void requestSpsFlowInfoCallback(shared_ptr<ICv2xTxFlow> txSpsFlow,
                                  const SpsFlowInfo & spsInfo,
                                  ErrorCode spsError) {
    if (ErrorCode::SUCCESS == spsError) {
        cout << "Priority: " << static_cast<int>(spsInfo.priority)
            << ", Periodicity: " << static_cast<int>(spsInfo.periodicity)
            << ", NbytesReserved: "<< spsInfo.nbytesReserved
            << ", Traffic class:"
            << static_cast<int>(Cv2xUtil::priorityToTrafficClass(spsInfo.priority)) << endl;
    }
    gCallbackPromise.set_value(spsError);
}

// Callback function for ICv2xRadio->requestDataSessionSettings()
static void requestDataSessionSettingsCallback(const DataSessionSettings & settings,
                                  ErrorCode spsError) {
    if (ErrorCode::SUCCESS == spsError) {
        if (settings.mtuValid) {
            cout << "MTU size: " << settings.mtu << endl;
        }
    }
    gCallbackPromise.set_value(spsError);
}

// Returns current timestamp
static uint64_t getCurrentTimestamp(void) {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec * 1000000ull + tv.tv_usec;
}

// Fills buffer with dummy data
static void fillBuffer(void) {

    static uint16_t seq_num = 0u;
    auto timestamp = getCurrentTimestamp();

    // Very first payload is test Magic number, this is  where V2X Family ID would normally be.
    gBuf[0] = TEST_VERNO_MAGIC;

    // Next byte is the UE equipment ID
    gBuf[1] = UEID;

    // Sequence number
    auto dataPtr = gBuf.data() + 2;
    uint16_t tmp = htons(seq_num++);
    memcpy(dataPtr, reinterpret_cast<char *>(&tmp), sizeof(uint16_t));
    dataPtr += sizeof(uint16_t);

    // Timestamp
    dataPtr += snprintf(dataPtr, G_BUF_LEN - (2 + sizeof(uint16_t)),
                        "<%llu> ", static_cast<long long unsigned>(timestamp));

    // Dummy payload
    constexpr int NUM_LETTERS = 26;
    auto i = 2 + sizeof(uint16_t) - sizeof(long long unsigned);
    for (; i < G_BUF_LEN; ++i) {
        gBuf[i] = 'a' + ((seq_num + i) % NUM_LETTERS);
    }
}

// Function for transmitting data
static void sampleTx(shared_ptr<ICv2xTxFlow> txFlow) {

    static uint32_t txCount = 0u;
    int sock = txFlow->getSock();

    cout << "sampleSpsTx(" << sock << ")" << endl;

    struct msghdr message = {0};
    struct iovec iov[1] = {0};
    struct cmsghdr * cmsghp = NULL;
    char control[CMSG_SPACE(sizeof(int))];

    // Send data using sendmsg to provide IPV6_TCLASS per packet
    iov[0].iov_base = gBuf.data();
    iov[0].iov_len = G_BUF_LEN;
    message.msg_iov = iov;
    message.msg_iovlen = 1;
    message.msg_control = control;
    message.msg_controllen = sizeof(control);

    // Fill ancillary data
    int priority = PRIORITY;
    cmsghp = CMSG_FIRSTHDR(&message);
    cmsghp->cmsg_level = IPPROTO_IPV6;
    cmsghp->cmsg_type = IPV6_TCLASS;
    cmsghp->cmsg_len = CMSG_LEN(sizeof(int));
    memcpy(CMSG_DATA(cmsghp), &priority, sizeof(int));

    // Send data
    auto bytes_sent = sendmsg(sock, &message, 0);
    cout << "bytes_sent=" << bytes_sent << endl;

    // Check bytes sent
    if (bytes_sent < 0) {
        cerr << "Error sending message: " << bytes_sent << endl;
        bytes_sent = -1;
    } else {
        if (bytes_sent == G_BUF_LEN) {
           ++txCount;
        } else {
            cerr << "Error : " << bytes_sent << " bytes sent." << endl;
        }
    }

    cout << "TX count: " << txCount << endl;
}

// Callback for ICv2xRadio->closeTxFlow()
static void closeFlowCallback(shared_ptr<ICv2xTxFlow> flow, ErrorCode error) {
    gCallbackPromise.set_value(error);
}

static void printUsage(const char *Opt) {
    cout << "Usage: " << Opt << "\n"
         << "-e event tx flow type\n"
         << "-r<auto-retrans mode>  0--disable 1--enable, default to enable\n" << endl;
}

// Parse options
static int parseOpts(int argc, char *argv[]) {
    int rc = 0;
    int c;
    while ((c = getopt(argc, argv, "?er:")) != -1) {
        switch (c) {
        case 'e':
            gFlowType = TxFlowType::EventOnly;
            cout << "Create Tx event flow" << endl;
            break;
        case 'r':
            if (optarg) {
                gAutoRetransMode = static_cast<bool>(atoi(optarg));
                cout << "auto retrans mode: " << static_cast<int>(gAutoRetransMode) << endl;
            }
            break;
        case '?':
        default:
            rc = -1;
            printUsage(argv[0]);
            return rc;
        }
    }

    return rc;
}

static int createTxFlow(void)
{
    if (gFlowType == TxFlowType::SpsOnly) {
        SpsFlowInfo spsInfo;
        spsInfo.priority = Priority::PRIORITY_2;
        spsInfo.periodicity = Periodicity::PERIODICITY_100MS;
        spsInfo.nbytesReserved = G_BUF_LEN;

        resetCallbackPromise();
        if(Status::SUCCESS != gCv2xRadio->createTxSpsFlow(TrafficIpType::TRAFFIC_NON_IP,
                                                         TX_SERVICE_ID,
                                                         spsInfo,
                                                         TX_SRC_PORT_NUM,
                                                         false,
                                                         0,
                                                         createSpsFlowCallback)
            || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
            cerr << "Failed to create tx sps flow" << endl;
            return EXIT_FAILURE;
        }

        resetCallbackPromise();
        if(Status::SUCCESS != gCv2xRadio->requestSpsFlowInfo(
                                    gTxFlow, requestSpsFlowInfoCallback)
            || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
            cerr << "Failed to request for sps flow info" << endl;
            return EXIT_FAILURE;
        }

        if(!gAutoRetransMode) {
            spsInfo.autoRetransEnabled = false;
            resetCallbackPromise();
            if(Status::SUCCESS != gCv2xRadio->changeSpsFlowInfo(gTxFlow,
                                                   spsInfo,
                                                   changeSpsFlowInfoCallback)
                || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
                cerr << "Failed to request to change sps flow info" << endl;
                return EXIT_FAILURE;
            }
        }
    } else if (gFlowType == TxFlowType::EventOnly) {
        EventFlowInfo eventInfo;

        resetCallbackPromise();
        if(Status::SUCCESS != gCv2xRadio->createTxEventFlow(TrafficIpType::TRAFFIC_NON_IP,
                                               TX_SERVICE_ID,
                                               eventInfo,
                                               TX_SRC_PORT_NUM,
                                               createEventFlowCallback)
            || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
            cerr << "Failed to create tx event flow" << endl;
            return EXIT_FAILURE;
        }

        if(!gAutoRetransMode) {
            eventInfo.autoRetransEnabled = false;
            resetCallbackPromise();
            if(Status::SUCCESS != gCv2xRadio->changeEventFlowInfo(gTxFlow,
                                                   eventInfo,
                                                   changeEventFlowInfoCallback)
                || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
                cerr << "Failed to request to change event flow info" << endl;
                return EXIT_FAILURE;
            }
        }
    } else {
        cerr << "Incorrect tx flow type." << endl;
        return EXIT_FAILURE;
    }

    cout << "TX flow: ipType= " << static_cast<int>(gTxFlow->getIpType())
        << ", ServiceId= " << gTxFlow->getServiceId()
        << ", PortNum= " << gTxFlow->getPortNum() << endl;
    return EXIT_SUCCESS;
}

int main(int argc, char *argv[]) {
    cout << "Running Sample C-V2X TX app" << endl;

    if (parseOpts(argc, argv) < 0) {
        return EXIT_FAILURE;
    }

    std::vector<std::string> groups{"system", "diag", "radio"};
    if (-1 == Utils::setSupplementaryGroups(groups)){
        cout << "Adding supplementary group failed!" << std::endl;
    }

    // Get handle to Cv2xRadioManager
    bool cv2xRadioManagerStatusUpdated = false;
    telux::common::ServiceStatus cv2xRadioManagerStatus =
            ServiceStatus::SERVICE_UNAVAILABLE;
    std::condition_variable cv;
    std::mutex mtx;
    auto statusCb = [&](ServiceStatus status) {
        std::lock_guard<std::mutex> lock(mtx);
        cv2xRadioManagerStatusUpdated = true;
        cv2xRadioManagerStatus = status;
        cv.notify_all();
    };
    // Get handle to Cv2xRadioManager
    auto & cv2xFactory = Cv2xFactory::getInstance();
    auto cv2xRadioManager = cv2xFactory.getCv2xRadioManager(statusCb);
    if (!cv2xRadioManager) {
        cout << "Error: failed to get Cv2xRadioManager." << endl;
        return EXIT_FAILURE;
    }
    std::unique_lock<std::mutex> lck(mtx);
    cv.wait(lck, [&] { return cv2xRadioManagerStatusUpdated; });
    if (ServiceStatus::SERVICE_AVAILABLE != cv2xRadioManagerStatus
        || ServiceStatus::SERVICE_AVAILABLE != cv2xRadioManager->getServiceStatus()) {
        cerr << "C-V2X Radio Manager initialization failed, exiting" << endl;
        return EXIT_FAILURE;
    }

    // Get C-V2X status and make sure Tx is enabled
    if(Status::SUCCESS != cv2xRadioManager->requestCv2xStatus(cv2xStatusCallback)
        || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
        cerr << "Failed to request for Cv2x status" << endl;
        return EXIT_FAILURE;
    }

    if (Cv2xStatusType::ACTIVE == gCv2xStatus.txStatus) {
        cout << "C-V2X TX status is active" << endl;
    }
    else {
        cerr << "C-V2X TX is inactive" << endl;
        return EXIT_FAILURE;
    }

    // Get handle to Cv2xRadio
    gCv2xRadio = cv2xRadioManager->getCv2xRadio(TrafficCategory::SAFETY_TYPE);

    // Wait for radio to complete initialization
    if (not gCv2xRadio->isReady()) {
        if (Status::SUCCESS == gCv2xRadio->onReady().get()) {
            cout << "C-V2X Radio is ready" << endl;
        }
        else {
            cerr << "C-V2X Radio initialization failed." << endl;
            return EXIT_FAILURE;
        }
    }

    resetCallbackPromise();
    if(Status::SUCCESS != gCv2xRadio->requestDataSessionSettings(requestDataSessionSettingsCallback)
        || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
        cerr << "Failed to request for data session settings" << endl;
        return EXIT_FAILURE;
    }

    if (EXIT_SUCCESS == createTxFlow()) {
        // Send message in a loop
        for (uint16_t i = 0; i < NUM_TEST_ITERATIONS; ++i) {
            fillBuffer();
            sampleTx(gTxFlow);
            usleep(100000u);
        }
    }

    // Deregister TX flow
    if (gTxFlow) {
        resetCallbackPromise();
        if(Status::SUCCESS != gCv2xRadio->closeTxFlow(gTxFlow, closeFlowCallback)
            || ErrorCode::SUCCESS != gCallbackPromise.get_future().get()) {
            cerr << "Failed to request to close tx flow" << endl;
            return EXIT_FAILURE;
        }
    }

    cout << "Done." << endl;

    return EXIT_SUCCESS;
}
